'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      scriptId: 'number',
      orderId: {
        type: 'string',
        required: true,
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: '订单号格式错误',
      },
      // userId: 'string',
      // appKey: 'string',
    };
    this.paramsRule = {
      type: {
        type: 'enum',
        required: true,
        values: ['gjj', 'shebao', 'incomeTax'],
      },
    };
    this.submitRule = {
      phone: { type: 'string', format: /^1\d{10}$/, required: false },
      ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false },
      email: { type: 'email', required: false },
    };

    this.fetchCaptureRule = {
      type: { type: 'enum', values: ['code', 'phone_code'] },
    };

    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
      type: {
        type: 'enum',
        required: true,
        values: ['gjj', 'shebao', 'incomeTax'],
      },
    };

    this.cityListRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { scriptId, appKey } = ctx.request.body;
    const { type } = ctx.params;
    await service.partner.checkInfo(appKey, type);
    const taskId = await service.task.create(scriptId, type);
    await service.cache.set({
      key: String(taskId),
      value: { status: 'init', note: { message: 'init' } },
      type,
    });
    ctx.success({ taskId });
  }

  async submit() {
    const { ctx, service, config } = this;
    const { key, iv } = config.aesParams;
    ctx.validate(this.submitRule);
    const { taskId, type } = ctx.params;
    const data = JSON.parse(JSON.stringify(ctx.request.body));
    if (data.hasOwnProperty('password')) {
      data.password = ctx.helper.aescipher('aes-128-cbc', key, iv, data.password);
    }
    if (data.hasOwnProperty('confirmPassword')) {
      data.confirmPassword = ctx.helper.aescipher('aes-128-cbc', key, iv, data.confirmPassword);
    }
    if (data.hasOwnProperty('name')) {
      data.name = ctx.helper.aescipher('aes-128-cbc', key, iv, data.name);
    }
    await service.cache.set({
      key: String(taskId),
      value: { status: 'login', note: { message: 'login' }, params: data },
      type,
    });
    await service.task.submit({
      taskId,
      data: ctx.request.body,
      callbackUrl: config.callbackUrl + type + '/callback',
    }, type);
    ctx.success({ taskId });
  }

  async fetchCapture() {
    const { ctx, service } = this;
    ctx.validate(this.fetchCaptureRule);
    const { taskId, type } = ctx.params;
    const { type: taskType } = ctx.request.body;
    const result = await service.task.fetchCapture({
      taskId,
      type,
      taskType,
    });
    ctx.success(result);
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.taskIdRule, ctx.params);
    const { taskId, type } = ctx.params;
    const data = await service.cache.get({ key: String(taskId), type });
    ctx.success(data);
  }

  async handleCallback() {
    const { ctx, service } = this;
    const { taskId } = ctx.request.body;
    const { type } = ctx.params;
    ctx.body = {
      code: '0',
    };
    const result = await service.task.fetchTask({ taskId: String(taskId), type });
    let taskNote = {
      status: 'init',
      note: { message: 'init' },
    };
    switch (result.code) {
    case -1:
    case 106:
    case 105:
      taskNote = {
        status: 'failure',
        note: { message: result.msg },
      };
      break;
    case 1:
      if (result.data.data) {
        result.data = {
          data: JSON.parse(JSON.parse(result.data.data).post_data.strForNextStep),
          loginParam: result.data.loginParam,
        };
      }
      taskNote = {
        status: 'next',
        note: { message: 'waiting', nextStep: result.data },
      };
      break;
    case 110:
      taskNote = {
        status: 'query',
        note: { message: 'login success' },
      };
      break;
    case 0:
      taskNote = {
        status: 'success',
        note: { message: 'task success' },
      };
      break;
    default:
      ctx.logger.warn('handleCallback', JSON.stringify(result));
      break;
    }
    await service.cache.set({
      key: String(taskId),
      value: taskNote,
      type,
    });
  }

  async fetchCityConfig() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.cityListRule);
      await service.signature.signatureCheck(ctx.request.body);
      const result = await service.scripts.getCityList(ctx.request.body);
      ctx.body = result;
      return;
    } catch (err) {
      ctx.logger.error('fetchCityConfig ERROR', JSON.stringify(err));
      ctx.error(err);
    }
  }
}

module.exports = TaskController;
