'use strict';

const Service = require('egg').Service;

class ScriptsService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config } = this;
    const { scriptsAPI } = config;
    this.scriptsAPI = scriptsAPI;
    this.baseURL = scriptsAPI.host;
    this.fetchScriptsUrl = scriptsAPI.fetchScriptsUrl;
    this.fetchOneScriptUrl = scriptsAPI.fetchOneScriptUrl;
    this.fetchParamsInfoUrl = scriptsAPI.fetchParamsInfoUrl;
    this.fetchHelpUrl = scriptsAPI.fetchHelpUrl;
    this.redisScriptsKey = scriptsAPI.redisScriptsKey;
    this.redisParamsKey = scriptsAPI.redisParamsKey;
    this.redisScriptListKey = scriptsAPI.redisScriptListKey;
    this.cityListUrl = scriptsAPI.cityListUrl;
  }

  checkType(type) {
    const { ctx } = this;
    let scriptsType = '';
    switch (type) {
    case 'gjj':
      scriptsType = '/hf';
      break;
    case 'shebao':
      break;
    case 'incomeTax':
      break;
    default:
      ctx.throw(422, { message: '无效的类型' });
      break;
    }
    return scriptsType;
  }

  async fetchScripts(type) {
    const { baseURL, fetchScriptsUrl, redisScriptsKey, ctx } = this;
    const data = await this.app.redis.get(redisScriptsKey + type);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Scripts】fetchScriptsFromRedis', 'result:', data, err);
        await this.app.redis.del(redisScriptsKey + type);
      }
    }
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchScriptsUrl, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchScripts,${baseURL + fetchScriptsUrl}`, 'result:', type, JSON.stringify(result.data));
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisScriptsKey + type, JSON.stringify(result.data), 'EX', 120);
    }
    return result.data;
  }

  async fetchOneScripts(scriptId, type) {
    const { baseURL, fetchOneScriptUrl, redisScriptListKey, ctx } = this;
    const data = await this.app.redis.get(redisScriptListKey + type + scriptId);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Scripts】fetchOneScriptsFromRedis', 'result:', data, err);
        await this.app.redis.del(redisScriptListKey + type + scriptId);
      }
    }
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchOneScriptUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    if (result.data && (String(result.data.id) === scriptId)) {
      await this.app.redis.set(redisScriptListKey + type + scriptId, JSON.stringify(result.data), 'EX', 120);
    }
    return result.data;
  }

  async fetchParams(refresh = false, type) {
    const { baseURL, fetchParamsInfoUrl, redisParamsKey, ctx } = this;
    if (!refresh) {
      const data = await this.app.redis.get(redisParamsKey + type);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchParams', 'result:', data, err);
          await this.app.redis.del(redisParamsKey + type);
        }
      }
    }

    const result = await ctx.curl(baseURL + this.checkType(type) + fetchParamsInfoUrl, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchParams,${baseURL + fetchParamsInfoUrl}`, 'result:', JSON.stringify(result.data));
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisParamsKey + type, JSON.stringify(result.data), 'EX', 300);
    }
    return result.data;
  }

  async fetchHelp(scriptId, type) {
    const { baseURL, scriptsAPI: { fetchHelpUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchHelpUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchQueryButton(scriptId, type) {
    const { baseURL, scriptsAPI: { fetchQueryButtonUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchQueryButtonUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchNotice(scriptId, type) {
    const { baseURL, scriptsAPI: { fetchNoticeUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchNoticeUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchHubSeripts(hubId, type) {
    const { baseURL, scriptsAPI: { fetchHubSeriptsUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchHubSeriptsUrl + '/' + hubId + '/scripts', {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }
  async fetchCityFormIp() {
    const { ctx } = this;
    const { ip } = ctx.request;
    ctx.logger.info(JSON.stringify(ctx.request));
    const ipArray = ip.split(',');
    const url = `https://api.map.baidu.com/location/ip?ak=3TBenWOhPygtFFazaR5kSibU&ip=${ipArray[0]}`;
    const result = await ctx.curl(url, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    if (result.data.status === 0) {
      return result.data.content.address_detail.city;
    }
    ctx.logger.error('fetchCityFormIp', url, JSON.stringify(result.data));
    return '北京市';
  }

  async fetchScriptByCityName(name, type) {
    const { baseURL, scriptsAPI: { fetchScriptByCityNameUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + this.checkType(type) + fetchScriptByCityNameUrl + '?name=' + encodeURIComponent(name) + '&type=query', {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`fetchScriptByCityName,${baseURL + fetchScriptByCityNameUrl + '?name=' + name}`, 'result:', JSON.stringify(result.data));

    return result.data;
  }

  async fetchScriptIdByIp(type) {
    const city = await this.fetchCityFormIp();
    const scriptList = await this.fetchScriptByCityName(city, type);
    return scriptList;
  }

  async fetchScriptName(scriptId) {
    try {
      const city = await this.fetchOneScripts(scriptId);
      return city.name || '未知';
    } catch (err) {
      return '未知';
    }
  }

  async getCityList(params) {
    try {
      const { cityListUrl, ctx } = this;
      const result = await ctx.curl(cityListUrl, {
        charset: 'utf-8',
        timeout: [30000, 30000],
        dataType: 'json',
        contentType: 'json',
        method: 'POST',
        data: params,
      });
      return result.data;
    } catch (err) {
      this.ctx.logger.error(err);
      return { code: -1, msg: '系统错误,请稍后重试' };
    }
  }
}

module.exports = ScriptsService;
