'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      scriptId: 'number',
      orderId: 'string',
    };

    this.submitRule = {
      orderSn: {
        type: 'string',
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: 'orderSn格式错误',
      },
      params: {
        type: 'object',
        rule: {
          phone: { type: 'string', format: /^1\d{10}$/, required: false, message: '手机号格式错误' },
          ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false, message: '身份证格式错误' },
        },
      },
    };

    this.fetchCaptureRule = {
      orderSn: {
        type: 'string',
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: 'orderSn格式错误',
      },
      type: { type: 'enum', values: ['code', 'phone_code'] },
    };

    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
    };

    this.cityListRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  // 创建任务接口
  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { scriptId } = ctx.request.body;
    const taskId = await service.task.create(scriptId);
    ctx.success({ taskId });
  }

  // 提交任务接口
  async submit() {
    const { ctx, service, config } = this;
    try {
      ctx.validate(this.submitRule);
      const { orderSn } = ctx.request.body;
      const order = await service.cache.get({ key: orderSn });
      if (order.status === 'success' || order.status === 'failure') {
        ctx.body = {
          code: -1,
          msg: '此任务已结束',
        };
        return;
      }
      await service.cache.set({
        key: orderSn, value: {
          status: 'submit',
          note: { code: 3, msg: '任务已提交', data: { data: null, loginParam: null, cityId: order.cityId, orderSn } },
        },
      });
      await service.task.submit({
        taskId: order.taskId,
        data: ctx.request.body.params,
        callbackUrl: config.callbackUrl,
      });
      ctx.body = {
        msg: null,
        code: 0,
        data: {
          cityId: order.cityId,
          orderSn,
        },
      };
    } catch (err) {
      ctx.logger.error('submit', err);
      ctx.status = 200;
      if (err.code === 'invalid_param') {
        ctx.body = { code: -1, msg: err.errors[0].message };
        return;
      }
      if (/[a-zA-Z]+/.test(err.message)) {
        ctx.body = { code: err.code || -1, msg: '系统错误, 请稍后再试' };
        return;
      }
      ctx.body = { code: err.code || -1, msg: err.message || '' };
    }
  }

  // 获取验证码接口
  async fetchCapture() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.fetchCaptureRule);
      const { type, orderSn } = ctx.request.body;
      const order = await service.cache.get({ key: orderSn });
      if (order.status === 'success' || order.status === 'failure') {
        ctx.body = {
          code: -1,
          msg: '此任务已结束',
        };
        return;
      }
      const result = await service.task.fetchCapture({
        taskId: order.taskId,
        type,
      });
      delete result.taskId;
      ctx.body = {
        code: 0,
        data: { ...result, orderSn },
      };
    } catch (err) {
      ctx.logger.error('submit', err);
      ctx.status = 200;
      if (err.code === 'invalid_param') {
        ctx.body = { code: -1, msg: '请求参数有误' };
        return;
      }
      if (/[a-zA-Z]+/.test(err.message)) {
        ctx.body = { code: err.code || -1, msg: '系统错误, 请稍后再试' };
        return;
      }
      ctx.body = { code: err.code || -1, msg: err.message || '' };
    }
  }

  // 任务回调接口
  async handleCallback() {
    const { ctx, service } = this;
    const { taskId } = ctx.request.body;
    const result = await service.task.fetchTask({ taskId });
    ctx.success();
    const order = await service.order.getOneByTaskId(taskId);
    if (!order) {
      ctx.logger.error('handleCallback', taskId, '未查询到此任务');
      return;
    }
    const { orderId, appKey, cityId } = order;
    let taskNote = {
      status: 'init',
      note: { code: 2, msg: '任务正在执行', data: { data: null, loginParam: null, cityId, orderSn: orderId } },
    };
    switch (result.code) {
    case -1:
    case 106:
    case 102:
    case 204:
      taskNote = {
        status: 'failure',
        note: { code: -1, msg: result.msg, data: { data: null, loginParam: null, cityId, orderSn: orderId } },
      };
      await service.cache.set({
        key: orderId,
        value: taskNote,
      });
      await service.partner.notice(order);
      break;
    case 1:
      taskNote = {
        status: 'next',
        note: { code: 1, msg: null, data: { data: null, loginParam: result.loginParam, cityId, orderSn: orderId } },
      };
      await service.cache.set({
        key: orderId,
        value: taskNote,
      });
      await service.partner.notice(order);
      break;
    case 110:
      taskNote = {
        status: 'query',
        note: { code: 110, msg: '任务提交成功', data: { data: null, loginParam: null, cityId, orderSn: orderId } },
      };
      break;
    case 0:
      taskNote = {
        status: 'success',
        note: { code: 0, msg: '任务成功', data: { data: null, loginParam: null, cityId, orderSn: orderId } },
      };
      try {
        const insertData = await service.washData.wash(result);
        insertData.orderId = orderId;
        insertData.cityName = await service.scripts.fetchScriptName(insertData.cid);
        insertData.taskId = Number(taskId);
        insertData.appKey = appKey;
        delete insertData.code;
        await service.storage.write(insertData);
        await service.cache.set({
          key: orderId,
          value: taskNote,
        });
        await service.partner.notice(order);
      } catch (err) {
        ctx.logger.error('handleCallback', err, JSON.stringify(result));
        taskNote = {
          status: 'failure',
          note: { code: -1, msg: '系统错误', data: { data: null, loginParam: null, cityId: order.cityId, orderSn: order.orderId } },
        };
      }
      break;
    default:
      ctx.logger.warn('handleCallback', JSON.stringify(result));
      break;
    }
  }

}

module.exports = TaskController;
