/*
 * @Descripttion:
 * @version:
 * @Author: jd
 * @Date: 2019-02-25 18:59:53
 * @LastEditors  : jd
 * @LastEditTime : 2019-12-24 14:17:15
 */
'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      scriptId: 'number',
      orderId: 'string',
    };

    this.submitRule = {
      phone: { type: 'string', format: /^1\d{10}$/, required: false },
      ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false },
      email: { type: 'email', required: false },
    };

    this.fetchCaptureRule = {
      type: { type: 'enum', values: ['code', 'phone_code'] },
    };

    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
    };

    this.cityListRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { scriptId, orderId } = ctx.request.body;
    await service.task.iplimit();
    const taskId = await service.task.create(scriptId);
    await service.order.update({ orderId, cityId: scriptId, taskId: String(taskId) });
    await service.cache.set({
      key: String(taskId),
      value: { status: 'init', note: { message: 'init' } },
    });
    ctx.success({ taskId });
  }

  async submit() {
    const { ctx, service, config } = this;
    ctx.validate(this.submitRule);
    const { taskId } = ctx.params;
    let key = taskId + ctx.request.body.timestamp;
    if (key.length < 16) {
      while (key.length < 16) key += '0';
    }
    key = key.substring(0, 16);
    ctx.request.body = JSON.parse(ctx.helper.aesDecrypt({ algorithm: 'aes-128-ecb', key, data: ctx.request.body.data }));
    await service.cache.set({
      key: String(taskId),
      value: { status: 'login', note: { message: 'login' } },
    });
    await service.task.submit({
      taskId,
      data: ctx.request.body,
      callbackUrl: config.callbackUrl,
    });
    ctx.success({ taskId });
  }

  async fetchCapture() {
    const { ctx, service } = this;
    ctx.validate(this.fetchCaptureRule);
    const { taskId } = ctx.params;
    const { type } = ctx.request.body;
    const result = await service.task.fetchCapture({
      taskId: String(taskId),
      type,
    });
    ctx.success(result);
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.taskIdRule, ctx.params);
    const { taskId } = ctx.params;
    const data = await service.cache.get({ key: String(taskId) });
    ctx.success(data);
  }

  async handleCallback() {
    const { ctx, service } = this;
    const { taskId } = ctx.request.body;
    ctx.body = {
      code: '0',
    };
    const result = await service.task.fetchTask({ taskId: String(taskId) });
    let taskNote = {
      status: 'init',
      note: { message: 'init' },
    };
    switch (result.code) {
      case -1:
      case 106:
      case 102:
      case 204:
        taskNote = {
          status: 'failure',
          note: { message: result.msg },
        };
        break;
      case 1:
        if (result.data.data) {
          result.data = {
            data: JSON.parse(JSON.parse(result.data.data).post_data.strForNextStep),
            loginParam: result.data.loginParam,
          };
        }
        taskNote = {
          status: 'next',
          note: { message: 'waiting', nextStep: result.data },
        };
        break;
      case 110:
        taskNote = {
          status: 'query',
          note: { message: 'login success' },
        };
        break;
      case 0:
        taskNote = {
          status: 'success',
          note: { message: 'task success' },
        };
        try {
          const insertData = await service.washData.wash(result);
          const order = await service.order.getOneByTaskId(String(taskId));
          if (!order) {
            throw new Error('任务已经结束了');
          }
          const { orderId, appKey } = order;
          insertData.orderId = orderId;
          insertData.cityName = await service.scripts.fetchScriptName(insertData.cid);
          insertData.taskId = taskId;
          insertData.appKey = appKey;
          delete insertData.code;
          await service.storage.write(insertData);
          await service.order.oldgjjStatus({ orderId, status: 'success' });
          await service.cache.set({
            key: String(taskId),
            value: taskNote,
          });
          await service.partner.notice(order);
        } catch (err) {
          ctx.logger.error('handleCallback', JSON.stringify(err), JSON.stringify(result));
          taskNote = {
            status: 'failure',
            note: { message: err.message },
          };
          if (!/token|appKey/.test(err.message)) {
            if (/[a-zA-Z]+/.test(err.message)) {
              taskNote.note = '系统错误, 请稍后再试';
            }
          }
        }
        break;
      default:
        ctx.logger.warn('handleCallback', JSON.stringify(result));
        break;
    }
    if (result.code !== 0) {
      await service.cache.set({
        key: String(taskId),
        value: taskNote,
      });
    }
  }

  async fetchCityConfig() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.cityListRule);
      await service.signature.signatureCheck(ctx.request.body);
      const result = await service.scripts.getCityList(ctx.request.body);
      ctx.body = result;
      return;
    } catch (err) {
      ctx.logger.error('fetchCityConfig ERROR', JSON.stringify(err));
      ctx.error(err);
    }
  }
}

module.exports = TaskController;
