'use strict';

const Controller = require('egg').Controller;

class OrderController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      appKey: 'string',
      userId: 'string',
      orderId: {
        type: 'string',
        required: false,
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: '订单号格式错误',
      }
    };

    this.showRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          orderId: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    const uuid = require('node-uuid');
    ctx.validate(this.createRule);
    let { appKey, userId, notifyUrl, backUrl, orderId } = ctx.request.body;
    orderId = orderId || uuid.v1();
    await service.order.create({
      orderId,
      taskId: '',
      userId: userId,
      cityId: '',
      notifyUrl: notifyUrl || (ctx.app.notifyMap.get(appKey) && ctx.app.notifyMap.get(appKey).notifyUrl) || '',
      backUrl: backUrl || '',
      appKey: appKey,
      status: 'init',
    });
    ctx.success({ orderId });
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.showRule);
    const { orderId } = ctx.params;
    await service.signature.signatureCheck(ctx.request.body);
    const data = await service.order.getOneByOrderId(orderId);
    if (data) {
      if (data.status === 'success') {
        const result = await service.storage.read(orderId);
        ctx.success(result);
        return;
      }
      ctx.success({
        status: data.status,
      });
      return;
    }
    ctx.fail('无此订单号');
  }

  async partnerShow() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.showRule);
      const { appKey, orderId } = ctx.request.body.params;
      await service.signature.signatureCheck(ctx.request.body);
      const data = await service.order.getOneByOrderId(orderId);
      if (data) {
        if (data.status === 'success') {
          const result = await service.storage.read(orderId, appKey);
          ctx.body = {
            code: 0,
            data: result
          };
          return;
        }
      }
      ctx.body = {
        code: -1,
        message: '无此订单号'
      };
    } catch (err) {
      ctx.status = 200;
      if (err.code == 'invalid_param') {
        return ctx.body = { code: -1, message: err.message || '' };
      }
      return ctx.body = { code: err.code || -1, message: err.message || '' };
    }
  }

  async orderShow() {
    const { ctx, service } = this;
    const { appKey, orderId } = ctx.params;
    const data = await service.order.getOneByOrderId(orderId);
    if (data) {
      if (data.status === 'success' && (data.appKey === appKey)) {
        const result = await service.storage.read(orderId, appKey);
        const ret = service.washData.dealData(result);
        ctx.success(ret);
        return;
      }
    }
    ctx.fail('无此订单号');
  }

  // async fetchOrderId() {
  //   const { ctx, service } = this;
  //   const signParams = ctx.helper.buildRequestBody({ token });
  //   const { appKey } = signParams.params;
  //   const orderId = await service.signature.createOrderId(signParams);
  //   await service.order.create({
  //     orderId,
  //     taskId: '',
  //     userId: '',
  //     cityId: '',
  //     appKey: appKey,
  //     status: 'init',
  //   });
  //   ctx.success({ orderId });
  // }
}

module.exports = OrderController;
