'use strict';

const Service = require('egg').Service;

const LOCK_SCRIPT = 'return redis.call("set", KEYS[1], ARGV[1], "NX", "EX", ARGV[2])';
const UNLOCK_SCRIPT =
    'if redis.call("get", KEYS[1]) == ARGV[1] then return redis.call("del", KEYS[1]) else return 0 end';

class SignatureService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config } = this;
    const { signatureAPI, lockKeys } = config;

    this.baseURL = signatureAPI.host;
    this.fetchTokenUrl = signatureAPI.fetchTokenUrl;
    this.fetchOrderIdUrl = signatureAPI.fetchOrderIdUrl;
    this.signatureUrl = signatureAPI.signatureUrl;
    this.signatureType = signatureAPI.signatureType;
    this.LOCK_KEY = lockKeys.fecteToken;
    this.SCRIPTS_KEY = lockKeys.token;

    ctx.app.redis.defineCommand('lock', {
      numberOfKeys: 1,
      lua: LOCK_SCRIPT,
    });
    ctx.app.redis.defineCommand('unlock', {
      numberOfKeys: 1,
      lua: UNLOCK_SCRIPT,
    });
  }

  _request(url, opts) {
    const { ctx, baseURL } = this;
    url = `${baseURL}${url}`;
    opts = {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      ...opts,
    };
    ctx.logger.info('signnature', url, opts);
    return ctx.curl(url, opts);
  }

  _checkSuccess(result) {
    if (result.status !== 200) {
      const errorMsg = result.data && result.data.error_msg ? result.data.error_msg : 'unknown error';
      this.ctx.throw(result.status, errorMsg);
    }
    if (result.data.code !== 0) {
      this.ctx.throw(400, { message: result.data.msg, code: result.data.code });
    }
  }

  async createToken(params) {
    const { fetchTokenUrl, ctx } = this;
    const result = await this._request(fetchTokenUrl, {
      method: 'post',
      data: params,
      contentType: 'json',
    });
    ctx.logger.info('【Signature】createToken params', params, 'result:', result.data);
    this._checkSuccess(result);
    return result.data.data.token;
  }

  async createOrderId(params) {
    const { fetchOrderIdUrl, ctx } = this;
    const result = await this._request(fetchOrderIdUrl, {
      method: 'post',
      data: params,
      contentType: 'json',
    });
    ctx.logger.info('【Signature】createOrderId params', params, 'result:', result.data);
    this._checkSuccess(result);
    return result.data.data.orderSn;
  }

  async signatureCheck(params) {
    const { signatureUrl, signatureType, ctx } = this;
    const result = await this._request(signatureUrl, {
      method: 'post',
      data: { ...params, type: signatureType },
      contentType: 'json',
    });
    ctx.logger.info('【Signature】signatureCheck params', params, 'result:', result.data);
    this._checkSuccess(result);
    return result.data.data;

  }

  // 临时文件
  async getToken() {
    const { app, ctx, LOCK_KEY, SCRIPTS_KEY } = this;
    const { redis } = app;
    const result = await redis.get(SCRIPTS_KEY);
    if (result) {
      return result;
    }
    const lockVal = ctx.helper.genRandomStr(16);
    const locked = await redis.lock(LOCK_KEY, lockVal, 60);
    if (locked) {
      const signParams = ctx.helper.buildRequestBody();
      const token = await this.createToken(signParams);
      await redis.set(SCRIPTS_KEY, token, 'EX', 7150);
      await redis.unlock(LOCK_KEY, lockVal);
      return token;
    }
    this.getToken();
  }

  async _checkSuccessForPartner(result) {
    if (result.status !== 200) {
      const errorMsg = result.data && result.data.error_msg ? result.data.error_msg : 'unknown error';
      this.ctx.throw(result.status, errorMsg);
    }
    if (result.data.code !== 0) {
      this.ctx.throw(200, { message: result.data.msg, code: result.data.code });
    }
  }

  // 给合作方创建token
  async createTokenForPartner(params) {
    const { fetchTokenUrl, ctx } = this;
    const result = await this._request(fetchTokenUrl, {
      method: 'post',
      data: params,
      contentType: 'json',
    });
    ctx.logger.info('【Signature】createToken params', params, 'result:', result.data);
    this._checkSuccessForPartner(result);
    return result.data.data.token;
  }
}

module.exports = SignatureService;
