'use strict';

const Service = require('egg').Service;

class PartnerService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config: { partnerAPI } } = this;
    this.partnerAPI = partnerAPI;
  }

  async fetchTheme(appKey) {
    const { ctx, partnerAPI: { fetchTheme, host, redisThemePrefix } } = this;
    if (appKey) {
      const data = await this.app.redis.get(redisThemePrefix + appKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchParams', redisThemePrefix + appKey, 'result:', data, err);
          await this.app.redis.del(redisThemePrefix + appKey);
        }
      }
    }
    const ret = await ctx.curl(`${host + fetchTheme}`, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      method: 'GET',
      data: {
        appKey,
        subject: 'hf',
      },
    });
    ctx.logger.info(`【Partner】 fetchTheme ${host + fetchTheme}/${appKey}`, ret.data);
    if (ret.status === 200 && ret.data && ret.data.length > 0 && ret.data[0].appKey === appKey) {
      await this.app.redis.set(redisThemePrefix + appKey, JSON.stringify(ret.data[0]), 'EX', 300);
    }
    return ret.data[0];
  }

  async fetchInfo(appKey) {
    const { ctx, partnerAPI: { fetchInfo, host, redisInfoPrefix } } = this;
    if (appKey) {
      const data = await this.app.redis.get(redisInfoPrefix + appKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchInfo', redisInfoPrefix + appKey, 'result:', data, err);
          await this.app.redis.del(redisInfoPrefix + appKey);
        }
      }
    }
    const ret = await ctx.curl(`${host + fetchInfo}`, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      method: 'GET',
      data: {
        appKey,
      },
    });
    ctx.logger.info(`【Partner】 fetchTheme ${host + fetchInfo}?appKey=${appKey}`, ret.data);
    if (ret.status === 200 && ret.data && ret.data.length > 0 && ret.data[0].appKey === appKey) {
      await this.app.redis.set(redisInfoPrefix + appKey, JSON.stringify(ret.data[0]), 'EX', 300);
    }
    return ret.data[0];
  }

  async fetchAgreements(id) {
    const { ctx, partnerAPI: { fetchAgreements, host, redisAgreementsPrefix } } = this;
    const data = await this.app.redis.get(redisAgreementsPrefix + id);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Partner】 fetchAgreements', redisAgreementsPrefix + id, 'result:', JSON.stringify(data), err);
        await this.app.redis.del(redisAgreementsPrefix + id);
      }
    }
    const url = `${host + fetchAgreements}/${id}`;
    const ret = await ctx.curl(url, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      method: 'GET',
    });
    ctx.logger.info(`【Partner】 fetchAgreements ${url}`, JSON.stringify(ret.data));
    if (ret.status === 200 && ret.data && ret.data.id === id) {
      await this.app.redis.set(redisAgreementsPrefix + id, JSON.stringify(ret.data), 'EX', 300);
    }
    return ret.data;
  }

  // 查询结束后回调通知合作方
  // 对于状态已经是成功的订单号不会在做通知
  // orderId/userId/status
  async notify({ taskId, cb, orderId, userId, appKey, status = 'success' }) {
    const { ctx, service } = this;
    if (!cb) {
      if (ctx.app.notifyMap.has(appKey)) {
        cb = ctx.app.notifyMap.get(appKey);
      } else {
        ctx.logger.info(`【orderId】:${orderId} 没有回调地址`);
        return;
      }
    }

    const data = await service.cache.get({ key: taskId });
    if (data && data.status === 'success') {
      ctx.logger.info(`【taskId】:${taskId} 模拟器多次通知任务成功情况`);
      return;
    }
    if (data) {
      try {
        await ctx.curl(cb, {
          method: 'POST',
          data: { orderSn: orderId, status, userId },
          dataType: 'json',
          contentType: 'json',
          timeout: ['30s', '30s'],
        });
        ctx.model.TaskStatus.update(
          { notice: 1 },
          { where: { taskId } }
        );
      } catch (e) { // 回调失败
        ctx.logger.error(`【cb】:${cb} 回调通知失败`);
        ctx.model.TaskStatus.update(
          { notice: 0 },
          { where: { taskId } }
        );
      }
    }
  }
}

module.exports = PartnerService;
