'use strict';

const Service = require('egg').Service;

const LOCK_SCRIPT = 'return redis.call("set", KEYS[1], ARGV[1], "NX", "EX", ARGV[2])';
const UNLOCK_SCRIPT =
    'if redis.call("get", KEYS[1]) == ARGV[1] then return redis.call("del", KEYS[1]) else return 0 end';

class TaskService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config: { taskAPI, LOCK_KEY } } = this;
    this.taskAPI = taskAPI;
    this.baseURL = taskAPI.host;
    this.LOCK_KEY = LOCK_KEY;
    ctx.app.redis.defineCommand('lock', {
      numberOfKeys: 1,
      lua: LOCK_SCRIPT,
    });
    ctx.app.redis.defineCommand('unlock', {
      numberOfKeys: 1,
      lua: UNLOCK_SCRIPT,
    });
  }


  _request(url, opts) {
    const { ctx, baseURL } = this;
    url = `${baseURL}${url}`;
    opts = {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      ...opts,
    };
    return ctx.curl(url, opts);
  }

  _checkSuccess(result) {
    if (result.status !== 200) {
      const errorMsg = result.data && result.data.error_msg ? result.data.error_msg : 'unknown error';
      this.ctx.throw(result.status, errorMsg);
    }
    if (result.data.code !== 0) {
      this.ctx.throw(400, { message: result.data.msg, code: result.data.code });
    }
  }

  async create(data) {
    const { ctx, taskAPI } = this;
    const { createTaskUrl } = taskAPI;
    const result = await this._request(createTaskUrl, {
      method: 'post',
      data: { cityId: data },
      contentType: 'json',
    });
    ctx.logger.info(`【Task】create ${createTaskUrl} cityId: ${data} result:`, result.data);
    this._checkSuccess(result);
    return result.data.data.taskId;
  }

  async fetchCapture({ taskId, type }) {
    const { taskAPI, ctx } = this;
    const { fetchCodeUrl } = taskAPI;
    const result = await this._request(fetchCodeUrl, {
      method: 'post',
      data: {
        taskId,
        type,
      },
      contentType: 'json',
    });
    ctx.logger.info(`【Task】fetchCapture ${fetchCodeUrl} params`, {
      taskId,
      type,
    }, result.data);
    this._checkSuccess(result);
    return result.data.data;
  }

  async submit(data) {
    const { taskAPI, ctx } = this;
    const { submitTaskUrl } = taskAPI;
    const result = await this._request(submitTaskUrl, {
      method: 'post',
      data,
      contentType: 'json',
    });
    ctx.logger.info(`【Task】submit ${submitTaskUrl} params`, data, result.data);
    this._checkSuccess(result);
    return result;
  }

  async fetchTask(data) {
    const { taskAPI, ctx } = this;
    const { fetchTaskUrl } = taskAPI;
    const result = await this._request(fetchTaskUrl, {
      method: 'post',
      data,
      contentType: 'json',
    });
    ctx.logger.info(`【Task】fetchTask ${fetchTaskUrl} params`, data, result.data);
    return result.data;
  }
  // async _fetchScripts() {
  //   const { taskAPI } = this;
  //   const { fecthHubsUrl } = taskAPI;
  //   const result = await this._request(fecthHubsUrl, {
  //     method: 'get',
  //     contentType: 'json',
  //   });
  //   return result;
  // }

  // async fetchScripts() {
  //   const { app, ctx, LOCK_KEY, SCRIPTS_KEY } = this;
  //   const { redis } = app;
  //   const result = await redis.get(SCRIPTS_KEY);
  //   if (result) {
  //     return JSON.parse(result);
  //   }
  //   const lockVal = ctx.helper.genRandomStr(16);
  //   const locked = await redis.lock(LOCK_KEY, lockVal, 60);
  //   if (locked) {
  //     const scripts = await this._fetchScripts();
  //     await redis.set(SCRIPTS_KEY, JSON.stringify(scripts), 'EX', 300);
  //     await redis.unlock(LOCK_KEY, lockVal);
  //     return scripts;
  //   }
  //   this.fetchScripts();
  // }


}

module.exports = TaskService;
