'use strict';

const Service = require('egg').Service;

class PartnerService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config: { partnerAPI } } = this;
    this.partnerAPI = partnerAPI;
  }

  async fetchTheme(appKey) {
    const { ctx, partnerAPI: { fetchTheme, host, redisThemePrefix } } = this;
    if (appKey) {
      const data = await this.app.cache.get(redisThemePrefix + appKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchParams', redisThemePrefix + appKey, 'result:', JSON.stringify(data), err);
          await this.app.cache.del(redisThemePrefix + appKey);
        }
      }
    }
    const ret = await ctx.curl(`${host + fetchTheme}/${appKey}`, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      method: 'GET',
    });
    ctx.logger.info(`【Partner】 fetchTheme ${host + fetchTheme}/${appKey}`, ret.data);
    if (ret.status === 200 && ret.data && ret.data.id === appKey) {
      await this.app.cache.set(redisThemePrefix + appKey, JSON.stringify(ret.data), 7200);
    }
    return ret.data;
  }

  async fetchScripts(appKey) {
    const { ctx, partnerAPI: { fetchScripts, host, redisScriptsPrefix } } = this;

    if (appKey) {
      const data = await this.app.cache.get(redisScriptsPrefix + appKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Partner】 fetchScripts', redisScriptsPrefix + appKey, 'result:', JSON.stringify(data), err);
          await this.app.cache.del(redisScriptsPrefix + appKey);
        }
      }
    }
    const url = `${host + fetchScripts}/${appKey}`;
    const ret = await ctx.curl(url, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      method: 'GET',
    });
    ctx.logger.info(`【Partner】 fetchTheme ${url}`, ret.data);
    if (ret.status === 200 && ret.data && ret.data.id === appKey) {
      await this.app.cache.set(redisScriptsPrefix + appKey, JSON.stringify(ret.data), 7200);
    }
    return ret.data;
  }

  // 查询结束后回调通知合作方
  // 对于状态已经是成功的订单号不会在做通知
  // orderId/userId/status
  async notify({ taskId, cb, orderId, userId, appKey }) {
    const { ctx, service } = this;
    if (!cb) {
      if (ctx.app.notifyMap.has(appKey)) {
        cb = ctx.app.notifyMap.get(appKey);
      } else {
        ctx.logger.info(`【orderId】:${orderId} 没有回调地址`);
        return;
      }
    }

    const data = await service.cache.get({ key: taskId });
    if (data && data.status === 'success') {
      ctx.logger.info(`【taskId】:${taskId} 模拟器多次通知任务成功情况`);
      return;
    }
    if (data) {
      try {
        await ctx.curl(cb, {
          method: 'POST',
          data: { orderId, status: 'success', userId },
          dataType: 'json',
          contentType: 'json',
          timeout: ['30s', '30s'],
        });
        ctx.model.TaskStatus.update(
          { notice: 1 },
          { where: { taskId } }
        );
      } catch (e) { // 回调失败
        ctx.logger.error(`【cb】:${cb} 回调通知失败`);
        ctx.model.TaskStatus.update(
          { notice: 0 },
          { where: { taskId } }
        );
      }
    }
  }

  async fetchAgreements(id) {
    const { ctx, partnerAPI: { fetchAgreements, host, redisAgreementsPrefix } } = this;
    const data = await this.app.cache.get(redisAgreementsPrefix + id);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Partner】 fetchAgreements', redisAgreementsPrefix + id, 'result:', JSON.stringify(data), err);
        await this.app.cache.del(redisAgreementsPrefix + id);
      }
    }
    const url = `${host + fetchAgreements}/${id}`;
    const ret = await ctx.curl(url, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      method: 'GET',
    });
    ctx.logger.info(`【Partner】 fetchAgreements ${url}`, JSON.stringify(ret.data));
    if (ret.status === 200 && ret.data && ret.data.id === id) {
      await this.app.cache.set(redisAgreementsPrefix + id, JSON.stringify(ret.data), 300);
    }
    return ret.data;
  }
}

module.exports = PartnerService;
