'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      scriptId: { type: 'number', required: true },
      orderId: { type: 'string', required: true },
      userId: { type: 'string', required: false },
    };
    this.submitRule = {
      phone: { type: 'string', format: /^1\d{10}$/, required: false },
      ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false },
      email: { type: 'email', required: false },
    };
    this.fetchCaptureRule = {
      type: { type: 'enum', values: ['code', 'phone_code'] },
    };
    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
    };
    this.centersInfoRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
    this.getCityConfig = {
      sign: 'string',
      type: { type: 'enum', values: ['shebao'] },
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { scriptId, orderId } = ctx.request.body;
    const taskId = await service.task.create(scriptId);
    await service.order.update({ orderId, cityId: scriptId, taskId });
    await service.cache.set({
      key: taskId,
      value: { status: 'init', note: { message: 'init' } },
    });
    ctx.success({ taskId });
  }

  async submit() {
    const { ctx, service, config } = this;
    ctx.validate(this.submitRule);
    const { taskId } = ctx.params;
    await service.task.submit({
      taskId,
      data: ctx.request.body,
      callbackUrl: config.callbackUrl,
    });
    await service.cache.set({
      key: taskId,
      value: { status: 'init', note: { message: 'init' } },
    });
    ctx.success(taskId);
  }

  async fetchCapture() {
    const { ctx, service } = this;
    ctx.validate(this.fetchCaptureRule);
    const { taskId } = ctx.params;
    const { type } = ctx.request.body;
    const result = await service.task.fetchCapture({
      taskId,
      type,
    });
    ctx.success(result);
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.taskIdRule, ctx.params);
    const { taskId } = ctx.params;
    const data = await service.cache.get({ key: taskId });
    ctx.success(data);
  }

  // 处理模拟器回调知
  async handleCallback() {
    const { ctx, service } = this;
    const { taskId } = ctx.request.body;
    const result = await service.task.fetchTask({ taskId });
    // ctx.success();
    // ctx.status = 200;
    ctx.body = 'ok';
    let taskNote = {
      status: 'init',
      note: { message: 'init' },
    };
    switch (result.code) {
    case -1:
    case 106:
    case 102:
    case 204:
      taskNote = {
        status: 'failure',
        note: { message: result.msg },
      };
      break;
    case 1:
      taskNote = {
        status: 'next',
        note: { message: 'waiting ', nextStep: result.data },
      };
      break;
    case 110:
      taskNote = {
        status: 'query',
        note: { message: 'login success' },
      };
      break;
    case 0:
      taskNote = {
        status: 'success',
        note: { message: 'task success' },
      };
      try {
        const insertData = await service.washData.wash(result);
        const order = await service.order.getOneByTaskId(taskId);
        if (!order) {
          throw new Error(`【taskId】:${taskId} 任务已经结束.`);
        }
        const { orderId, appKey, cityId } = order;
        insertData.orderId = orderId;
        // insertData.cityName = result.data.jsName.split('.')[0];
        insertData.jsName = result.data.jsName;
        insertData.cityName = ctx.app.cityNameMap.get(cityId);
        // TODO insertData.cityId -> cityName
        insertData.taskId = taskId;
        insertData.appKey = appKey;
        delete insertData.code;
        await service.storage.write(insertData);
        await service.order.oldsbStatus({ orderId, status: 'success' });
        await service.partner.notify(order);
      } catch (err) {
        ctx.logger.error(`【controller/task/handleCallback err】:${err}${JSON.stringify(result)}`);
        taskNote = {
          status: 'failure',
          note: { message: err.message },
        };
      }
      break;
    default:
      ctx.logger.error('【handleCallback】:', JSON.stringify(result));
      break;
    }
    await service.cache.set({
      key: taskId,
      value: taskNote,
    });
  }

  async fetchCentersInfo() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.centersInfoRule);
      const result = await service.scripts.getCentersInfo(ctx.request.body);
      ctx.body = result;
      return;
    } catch (err) {
      ctx.status = 200;
      if (err.code === 'invalid_param') {
        ctx.body = { code: -1, msg: '请求参数有误' };
        return;
      }
      if (/[a-zA-Z]+/.test(err.message)) {
        ctx.body = { code: err.code || -1, msg: '系统错误, 请稍后再试' };
        return;
      }
      ctx.body = { code: err.code || -1, msg: err.message || '' };
    }
  }

  async fetchCityConfig() {
    const { ctx, service } = this;
    try {
      const result = await service.scripts.getCityInfo();
      ctx.body = result;
      return;
    } catch (err) {
      ctx.status = 200;
      if (err.code === 'invalid_param') {
        ctx.body = { code: -1, msg: '请求参数有误' };
        return;
      }
      if (/[a-zA-Z]+/.test(err.message)) {
        ctx.body = { code: err.code || -1, msg: '系统错误, 请稍后再试' };
        return;
      }
      ctx.body = { code: err.code || -1, msg: err.message || '' };
    }
  }
}

module.exports = TaskController;
