'use strict';


const NODE_CACHE = Symbol('Context#NodeCache');

const NodeCache = require('node-cache');

class Cache {
  constructor(app) {
    this.app = app;
    this.name = 'unknown-cache';
  }

  async val(key, next, ttl) {
    let data = await this.get(key);
    if (data) {
      return data;
    }
    data = await next(key);
    await this.set(key, data, ttl);
    return data;
  }

  async get(key) {
    const startTime = +new Date();
    const ret = await this._get(key);
    let jsonText = JSON.stringify(ret);
    if (jsonText === undefined) {
      jsonText = 'undefined';
    } else if (jsonText.length >= 128) {
      if (/^\{/.test(jsonText)) {
        jsonText = '{...}';
      } else if (/^\[/.test(jsonText)) {
        jsonText = '[...]';
      } else {
        jsonText = jsonText.substr(0, 125) + '...';
      }
    }
    this.app.logger.info(`【Cache Get:】(${+new Date() - startTime}ms) ${this.name}.${key}: ${jsonText}`);
    return ret;
  }
  async set(key, value, ttl = 60) {
    const startTime = +new Date();
    await this._set(key, value, ttl);
    this.app.logger.info(`【Cache Set】(${+new Date() - startTime}ms) Key: ${key} value: ${JSON.stringify(value)}`);
    return;
  }

  async del(key) {
    if (key) {
      return await this._del(key);
    }
  }
}

class NodeCacheWrap extends Cache {
  constructor(app) {
    super(app);
    this.cache = new NodeCache();
    this.name = 'node-cache';
  }

  async _get(key) {
    return this.cache.get(key);
  }

  async _set(key, value, ttl) {
    const { cache } = this;
    value = JSON.parse(JSON.stringify(value));
    if (ttl > 0) {
      cache.set(key, value, ttl);
    } else {
      cache.set(key, value);
    }
  }

  async _del(key) {
    return this.cache.del(key);
  }
}

module.exports = {
  get cache() {
    if (!this[NODE_CACHE]) {
      this[NODE_CACHE] = new NodeCacheWrap(this);
    }
    return this[NODE_CACHE];
  },
};

