'use strict';

const Service = require('egg').Service;

class ScriptsService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config } = this;
    const { scriptsAPI } = config;
    this.scriptsAPI = scriptsAPI;
    this.baseURL = scriptsAPI.host;
    this.fetchScriptsUrl = scriptsAPI.fetchScriptsUrl;
    this.fetchOneScriptUrl = scriptsAPI.fetchOneScriptUrl;
    this.fetchParamsInfoUrl = scriptsAPI.fetchParamsInfoUrl;
    this.fetchHelpUrl = scriptsAPI.fetchHelpUrl;
    this.redisScriptsKey = scriptsAPI.redisScriptsKey;
    this.redisParamsKey = scriptsAPI.redisParamsKey;
  }

  async fetchScripts() {
    const { baseURL, fetchScriptsUrl, redisScriptsKey, ctx } = this;
    const data = await this.app.redis.get(redisScriptsKey);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Scripts】fetchScriptsFromRedis', 'result:', data, err);
        await this.app.redis.del(redisScriptsKey);
      }
    }
    const result = await ctx.curl(baseURL + fetchScriptsUrl, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchScripts,${baseURL + fetchScriptsUrl}`, 'result:', result.data);
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisScriptsKey, JSON.stringify(result.data), 'EX', 300);
    }

    return result.data;
  }
  async fetchOneScripts(scriptId) {
    const { baseURL, fetchOneScriptUrl, ctx } = this;
    const result = await ctx.curl(baseURL + fetchOneScriptUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchParams(refresh = false) {
    const { baseURL, fetchParamsInfoUrl, redisParamsKey, ctx } = this;
    if (!refresh) {
      const data = await this.app.redis.get(redisParamsKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchParams', 'result:', data, err);
          await this.app.redis.del(redisParamsKey);
        }
      }
    }

    const result = await ctx.curl(baseURL + fetchParamsInfoUrl, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchParams,${baseURL + fetchParamsInfoUrl}`, 'result:', result.data);
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisParamsKey, JSON.stringify(result.data), 'EX', 3000);
    }
    return result.data;
  }

  async fetchHelp(scriptId) {
    const { baseURL, scriptsAPI: { fetchHelpUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchHelpUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchQueryButton(scriptId) {
    const { baseURL, scriptsAPI: { fetchQueryButtonUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchQueryButtonUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchNotice(scriptId) {
    const { baseURL, scriptsAPI: { fetchNoticeUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchNoticeUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchHubSeripts(hubId) {
    const { baseURL, scriptsAPI: { fetchHubSeriptsUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchHubSeriptsUrl + '/' + hubId + '/scripts', {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }
  async fetchCityFormIp() {
    const { ctx } = this;
    const { ip } = ctx.request;
    ctx.logger.info(ctx.request);
    const url = `https://api.map.baidu.com/location/ip?ak=3TBenWOhPygtFFazaR5kSibU&ip=${ip}`;
    const result = await ctx.curl(url, {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    if (result.data.status === 0) {
      return result.data.content.address_detail.city;
    }
    ctx.logger.error('fetchCityFormIp', url, result.data);
    return '北京市';
  }

  async fetchScriptByCityName(name) {
    const { baseURL, scriptsAPI: { fetchScriptByCityNameUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchScriptByCityNameUrl + '?name=' + encodeURIComponent(name) + '&type=query', {
      charset: 'utf-8',
      timeout: ['30s', '30s'],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`fetchScriptByCityName,${baseURL + fetchScriptByCityNameUrl + '?name=' + name}`, 'result:', result.data);

    return result.data;
  }

  async fetchScriptIdByIp() {
    const city = await this.fetchCityFormIp();
    const scriptList = await this.fetchScriptByCityName(city);
    return scriptList;
  }
}

module.exports = ScriptsService;
