'use strict';

const Controller = require('egg').Controller;

class OrderController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      appKey: { type: 'string', required: true },
      userId: { type: 'string', required: false },
      notifyUrl: { type: 'string', required: false },
      backUrl: { type: 'string', required: false },
      orderId: {
        type: 'string',
        required: false,
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: '订单号格式错误',
      },
    };
    this.showRule = {
      sign: 'string',
      params: {
        type: 'object', required: true,
        rule: {
          token: { type: 'string', required: true },
          orderId: { type: 'string', required: false },
          orderSn: { type: 'string', required: false }, // 老H5
          appKey: { type: 'string', required: true },
          timestamp: { type: 'string', required: true },
        },
      },
    };
    this.fetchRule = {
      sign: { type: 'string', required: true },
      params: {
        type: 'object', required: true, rule: {
          token: { type: 'string', required: true },
          appKey: { type: 'string', required: true },
          timestamp: { type: 'string', required: true },
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { appKey, userId, notifyUrl, backUrl } = ctx.request.body;
    let { orderId } = ctx.request.body;
    // const orderId = await service.signature.createOrderId(ctx.request.body);
    const toDbObj = {
      orderId,
      taskId: '',
      userId: userId || '',
      cityId: '',
      appKey,
      status: 'init',
      notifyUrl: notifyUrl || (ctx.app.notifyMap.get(appKey) && ctx.app.notifyMap.get(appKey).notifyUrl) || '',
      backUrl,
      type: 'h5',
    };
    if (orderId) { // 老的H5
      await service.order.update(toDbObj);
    } else {
      orderId = ctx.helper.getUuid();
      toDbObj.orderId = orderId;
      await service.order.create(toDbObj);
    }
    ctx.success({ orderId });
  }

  // 合作方获取订单数据
  async partnerShow() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.showRule);
      let { appKey, orderId, orderSn } = ctx.request.body.params;
      orderId = orderId || orderSn;
      const appKeyInfo = await service.partner.fetchInfo(appKey);
      if (!(appKeyInfo && appKeyInfo.valid === true && appKeyInfo.enableSsData === true)) {
        return ctx.fail('未开通此服务');
      }
      await service.signature.signatureCheck(ctx.request.body);
      const data = await service.order.getOneByOrderId({ orderId, status: 'success' });
      if (data) {
        if (data.status === 'success') {
          const result = await service.storage.read(orderId, appKey);
          ctx.body = {
            code: 0,
            data: result,
          };
          return;
        }
      }
      ctx.body = {
        code: -1,
        msg: '无此订单号',
      };
    } catch (err) {
      ctx.logger.error('partnerShow', JSON.stringify(err));
      ctx.status = 200;
      if (err.code === 'invalid_param') {
        ctx.body = { code: -1, msg: err.message || '' };
        return;
      }
      ctx.body = { code: err.code || -1, msg: err.message || '' };
      return;
    }
  }

  async fetchOrderId() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.fetchRule, ctx.request.body);
      const { appKey } = ctx.request.body.params;
      const orderId = await service.signature.createOrderId(ctx.request.body);
      await service.order.create({
        orderId,
        taskId: '',
        cityId: '',
        notifyUrl: ctx.app.notifyMap.get(appKey) && ctx.app.notifyMap.get(appKey).notifyUrl || '',
        appKey,
        status: 'init',
        type: 'h5',
      });
      ctx.body = {
        code: 0,
        data: {
          orderSn: orderId,
        },
        msg: '',
      };
    } catch (e) {
      ctx.logger.error('【fetchOrderId】:', JSON.stringify(e));
      ctx.status = 200;
      if (e.code === 'invalid_param') {
        ctx.body = { code: -1, msg: e.message || '' };
        return;
      }
      ctx.body = { code: e.code || -1, msg: e.message || '' };
      return;
    }
  }

  async orderShow() {
    const { ctx, service } = this;
    const { appKey, orderId } = ctx.params;
    const appKeyInfo = await service.partner.fetchInfo(appKey);
    if (!(appKeyInfo && appKeyInfo.valid === true && appKeyInfo.enableSsView === true)) {
      return ctx.fail('未开通此服务');
    }
    const data = await service.order.getOneByOrderId({ orderId });
    if (data) {
      if (data.status === 'success' && (data.appKey === appKey)) {
        const result = await service.storage.read(orderId, appKey);
        const ret = service.washData.dealData(result);
        ctx.success(ret);
        return;
      }
    }
    ctx.fail('无此订单号');
  }
}

module.exports = OrderController;
