'use strict';

const Service = require('egg').Service;

class ScriptsService extends Service {
    constructor(ctx) {
        super(ctx);
        const {config} = this;
        const {scriptsAPI, taskAPI} = config;
        this.scriptsAPI = scriptsAPI;
        this.baseURL = scriptsAPI.host;
        this.fetchScriptsUrl = scriptsAPI.fetchScriptsUrl;
        this.fetchOneScriptUrl = scriptsAPI.fetchOneScriptUrl;
        this.fetchParamsInfoUrl = scriptsAPI.fetchParamsInfoUrl;
        this.fetchHelpUrl = scriptsAPI.fetchHelpUrl;
        this.redisScriptsKey = scriptsAPI.redisScriptsKey;
        this.redisParamsKey = scriptsAPI.redisParamsKey;
        this.redisScriptListKey = scriptsAPI.redisScriptListKey;
        this.newhost = taskAPI.newhost;
        this.cityConfigUrl = taskAPI.cityConfigUrl;
    }

    async fetchScripts() {
        const {baseURL, fetchScriptsUrl, redisScriptsKey, newhost, cityConfigUrl, ctx} = this;
        const data = await this.app.redis.get(redisScriptsKey);
        if (data) {
            try {
                return JSON.parse(data);
            } catch (err) {
                ctx.logger.error('【Scripts】fetchScriptsFromRedis', 'result:', data, err);
                await this.app.redis.del(redisScriptsKey);
            }
        }

        let cityLists = [];
        let newCityMap = new Map();
        //个税管家
        const gsgjRet = await ctx.curl(newhost + cityConfigUrl, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        ctx.logger.info(`【Scripts】fetchScripts,gsgjRet,${baseURL + fetchScriptsUrl}`, 'result:', gsgjRet.data[1]);
        if (gsgjRet.data && gsgjRet.data.code === 0) {
            gsgjRet.data.data.map(Prov => {
                Prov.citys.map(city => {
                    let state = city.state === 1 ? 'online' : 'offline';
                    let obj = {
                        id: city.id,
                        state: state,
                        hub_name: city.name,
                        hub_id: city.id,
                        province_name: Prov.provinceName,
                        province_id: city.province,
                    };
                    newCityMap.set(String(city.id), obj)
                })
            });
        }

        //爬虫
        const result = await ctx.curl(baseURL + fetchScriptsUrl, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        ctx.logger.info(`【Scripts】fetchScripts,result,${baseURL + fetchScriptsUrl}`, 'result:', result.data[1]);
        if (result.data && result.data.length > 0) {
            result.data.map(city => {
                if (city.state === 'offline' && newCityMap.get(String(city.id)) && newCityMap.get(String(city.id)).state === 'online') {
                    city = newCityMap.get(String(city.id));
                    city.channelType = "gsgj";
                    ctx.logger.info(city.id, city.hub_name)
                } else {
                    city.channelType = "51gs";
                }
                cityLists.push(city)
            });
            await this.app.redis.set(redisScriptsKey, JSON.stringify(cityLists), 'EX', 300);
        }
        return cityLists;
    }

    async fetchOneScripts(scriptId) {
        const {baseURL, fetchOneScriptUrl, redisScriptListKey, newhost, cityConfigUrl, ctx} = this;
        const data = await this.app.redis.get(redisScriptListKey + scriptId);
        if (data) {
            try {
                return JSON.parse(data);
            } catch (err) {
                ctx.logger.error('【Scripts】fetchOneScriptsFromRedis', 'result:', data, err);
                await this.app.redis.del(redisScriptListKey + scriptId);
            }
        }
        let scriptData;
        const result = await ctx.curl(baseURL + fetchOneScriptUrl + '/' + scriptId, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        if (result.data) {
            if (result.data.state != 'online') {
                //个税管家
                // let newCityMap = new Map();
                const gsgjRet = await ctx.curl(newhost + cityConfigUrl, {
                    charset: 'utf-8',
                    timeout: ['30s', '30s'],
                    dataType: 'json',
                    contentType: 'json',
                });
                if (gsgjRet.data && gsgjRet.data.code === 0) {
                    gsgjRet.data.data.map(Prov => {
                        Prov.citys.map(city => {
                            if (city.state === 1) {
                                result.data.params = [];
                                for (let item of queryParam) {
                                    let objs = {
                                        login_type_id: '',
                                        name: '',
                                        login_step_detail: [{
                                            current_step: '',
                                            login_param: [],
                                            next_step: '',
                                            step_name: '',
                                            pre_step: ''
                                        }],
                                    };
                                    for (let gjParams of item.loginParam) {
                                        let gjObj = {};
                                        if (gjParams.type === 'radio') {
                                            gjObj = {
                                                name: gjParams.name,
                                                note: '',
                                                placeholder: gjParams.note,
                                                type: 'radio',
                                            }
                                        } else {
                                            gjObj = {
                                                name: gjParams.name,
                                                note: gjParams.note,
                                                placeholder: '',
                                                type: gjParams.type,
                                            }
                                        }
                                        objs.login_step_detail.login_param.push(gjObj)
                                    }
                                    result.data.params.push(objs)
                                }
                            }
                        })
                    });
                }
            } else {
                scriptData = result.data;
            }

            await this.app.redis.set(redisScriptListKey + scriptId, JSON.stringify(scriptData), 'EX', 60);
        }
        return scriptData;
    }

    async fetchParams(refresh = false) {
        const {baseURL, fetchParamsInfoUrl, redisParamsKey, ctx} = this;
        if (!refresh) {
            const data = await this.app.redis.get(redisParamsKey);
            if (data) {
                try {
                    return JSON.parse(data);
                } catch (err) {
                    ctx.logger.error('【Scripts】fetchParams', 'result:', data, err);
                    await this.app.redis.del(redisParamsKey);
                }
            }
        }

        const result = await ctx.curl(baseURL + fetchParamsInfoUrl, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        ctx.logger.info(`【Scripts】fetchParams,${baseURL + fetchParamsInfoUrl}`, 'result:', result.data);
        if (result.data && result.data.length > 0) {
            await this.app.redis.set(redisParamsKey, JSON.stringify(result.data), 'EX', 300);
        }
        return result.data;
    }

    async fetchHelp(scriptId) {
        const {baseURL, scriptsAPI: {fetchHelpUrl}, ctx} = this;
        const result = await ctx.curl(baseURL + fetchHelpUrl + '/' + scriptId, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        return result.data;
    }

    async fetchQueryButton(scriptId) {
        const {baseURL, scriptsAPI: {fetchQueryButtonUrl}, ctx} = this;
        const result = await ctx.curl(baseURL + fetchQueryButtonUrl + '/' + scriptId, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        return result.data;
    }

    async fetchNotice(scriptId) {
        const {baseURL, scriptsAPI: {fetchNoticeUrl}, ctx} = this;
        const result = await ctx.curl(baseURL + fetchNoticeUrl + '/' + scriptId, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        return result.data;
    }

    async fetchHubSeripts(hubId) {
        const {baseURL, scriptsAPI: {fetchHubSeriptsUrl}, ctx} = this;
        const result = await ctx.curl(baseURL + fetchHubSeriptsUrl + '/' + hubId + '/scripts', {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        return result.data;
    }

    async fetchCityFormIp() {
        const {ctx} = this;
        const {ip} = ctx.request;
        ctx.logger.info(ctx.request);
        const url = `https://api.map.baidu.com/location/ip?ak=3TBenWOhPygtFFazaR5kSibU&ip=${ip}`;
        const result = await ctx.curl(url, {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        if (result.data.status === 0) {
            return result.data.content.address_detail.city;
        }
        ctx.logger.error('fetchCityFormIp', url, JSON.stringify(result.data));
        return '北京市';
    }

    async fetchScriptByCityName(name) {
        const {baseURL, scriptsAPI: {fetchScriptByCityNameUrl}, ctx} = this;
        const result = await ctx.curl(baseURL + fetchScriptByCityNameUrl + '?name=' + encodeURIComponent(name) + '&type=query', {
            charset: 'utf-8',
            timeout: ['30s', '30s'],
            dataType: 'json',
            contentType: 'json',
        });
        ctx.logger.info(`fetchScriptByCityName,${baseURL + fetchScriptByCityNameUrl + '?name=' + name}`, 'result:', JSON.stringify(result.data));

        return result.data;
    }

    async fetchScriptIdByIp() {
        const city = await this.fetchCityFormIp();
        const scriptList = await this.fetchScriptByCityName(city);
        return scriptList;
    }

    async fetchScriptName(scriptId) {
        try {
            const city = await this.fetchOneScripts(scriptId);
            return city.name || '未知';
        } catch (err) {
            return '未知';
        }
    }
}

module.exports = ScriptsService;