'use strict';


const REDIS_CACHE = Symbol('Context#RedisCache');
const NODE_CACHE = Symbol('Context#NodeCache');

const NodeCache = require('node-cache');

class Cache {
    constructor(app) {
        this.app = app;
        this.name = 'unknown-cache';
    }

    async val(key, next, ttl) { // key存在取值，不存在存值
        let data = await this.get(key);
        if (data) {
            return data;
        }
        data = await next(key);
        await this.set(key, data, ttl);
        return data;
    }

    async get(key) {
        const startTime = +new Date();
        const ret = await this._get(key);
        let jsonText = JSON.stringify(ret);
        if (jsonText === undefined) {
            jsonText = 'undefined';
        } else if (jsonText.length >= 128) {
            if (/^\{/.test(jsonText)) {
                jsonText = '{...}';
            } else if (/^\[/.test(jsonText)) {
                jsonText = '[...]';
            } else {
                jsonText = jsonText.substr(0, 125) + '...';
            }
        }
        this.app.logger.info(`[cache](${+new Date() - startTime}ms) ${this.name}.${key}: ${jsonText}`);
        // this.cacheLog(startTime, key, jsonText);
        return ret;
    }
    async set(key, value, ttl = 60) {
        return await this._set(key, value, ttl);
    }

    async hget(key) {
        const startTime = +new Date();
        let ret;
        if (!this.hasOwnProperty('_hget') && this.hasOwnProperty('_get')) {
            ret = await this._get(key);
            return ret;
        }
        ret = await this._hget(key);
        this.chcheLog(startTime, key, ret);
        return ret;
    }

    async hset(key, obj, ttl = 60) {
        if (!this.hasOwnProperty('_hset') && this.hasOwnProperty('_set')) {
            return await this._set(key, obj, ttl);
        }
        return await this._hset(key, obj, ttl);
    }

    chcheLog(startTime, key, value) {
        if (typeof value === 'object') {
            value = JSON.stringify(value);
        }
        this.app.logger.info(`[cache](${+new Date() - startTime}ms) ${this.name}.${key}: ${value}`);
    }
}

class RedisCacheWrap extends Cache {
    constructor(app, redis) {
        super(app);
        this.redis = redis;
        this.name = 'redis-cache';
    }

    async _get(key) {
        const { redis } = this;
        let value = await redis.get(key);
        if (value === null) {
            value = undefined;
        }
        if (value) {
            value = JSON.parse(value);
        }
        return value;
    }
    async _set(key, value, ttl) {
        const { redis } = this;
        value = JSON.stringify(value);
        if (ttl > 0) {
            await redis.set(key, value, 'EX', ttl);
        } else {
            await redis.set(key, value);
        }
    }
    async _hget(key) {
        const { redis } = this;
        let value = await redis.hgetall(key);
        if (value === null) {
            value = undefined;
        }
        return value;
    }
    async _hset(key, obj, ttl) {
        const { redis } = this;
        if (typeof obj !== 'object') {
            obj = JSON.parse(obj);
        }
        if (ttl > 0) {
            await redis.hmset(key, obj);
        } else {
            await redis.hmset(key, obj);
        }
    }
}

class NodeCacheWrap extends Cache {
    constructor(app) {
        super(app);
        this.cache = new NodeCache();
        this.name = 'node-cache';
    }

    async _get(key) {
        return this.cache.get(key);
    }
    async _set(key, value, ttl) {
        const { cache } = this;
        value = JSON.parse(JSON.stringify(value));
        if (ttl > 0) {
            cache.set(key, value, ttl);
        } else {
            cache.set(key, value);
        }
    }
}

module.exports = {
    get memcache() {
        if (!this[REDIS_CACHE]) {
            this[REDIS_CACHE] = new RedisCacheWrap(this, this.redis);
        }
        return this[REDIS_CACHE];
    },
    get cache() {
        if (!this[NODE_CACHE]) {
            this[NODE_CACHE] = new NodeCacheWrap(this);
        }
        return this[NODE_CACHE];
    },
};