'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
    constructor(ctx) {
        super(ctx);
        this.createRule = {
            orderId: 'string',
        };

        this.submitRule = {
            phone: {type: 'string', format: /^1\d{10}$/, required: false},
            ID: {type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false},
            email: {type: 'email', required: false},
        };

        this.fetchCaptureRule = {
            type: {type: 'enum', values: ['code', 'phone_code']},
        };

        this.taskIdRule = {
            taskId: {type: 'string', format: /\d+/},
        };

        this.cityConfigRule = {
            sign: 'string',
            params: {
                type: 'object',
                rule: {
                    appKey: 'string',
                    timestamp: 'string',
                    token: 'string',
                }
            }
        }
    }

    async create() {
        const {ctx, service} = this;
        ctx.validate(this.createRule);
        const {orderId} = ctx.request.body;
        const taskId = await service.task.create();
        await service.order.update({orderId, taskId});
        await service.cache.set({
            key: taskId,
            value: {status: 'init', note: {message: 'init'}},
        });
        ctx.success({taskId});
    }

    async submit() {
        const {ctx, service, config} = this;
        ctx.validate(this.submitRule);
        const {taskId} = ctx.params;
        await service.cache.set({
            key: taskId,
            value: {status: 'init', note: {message: 'init'}},
        });
        await service.task.submit({
            taskId,
            data: ctx.request.body,
            callbackUrl: config.callbackUrl,
        });
        ctx.success({taskId});
    }

    async fetchCapture() {
        const {ctx, service} = this;
        ctx.validate(this.fetchCaptureRule);
        const {taskId} = ctx.params;
        const {type} = ctx.request.body;
        const result = await service.task.fetchCapture({
            taskId,
            type,
        });
        ctx.success(result);
    }

    async show() {
        const {ctx, service} = this;
        ctx.validate(this.taskIdRule, ctx.params);
        const {taskId} = ctx.params;
        const data = await service.cache.get({key: taskId});
        ctx.success(data);
    }

    async handleCallback() {
        const {ctx, service} = this;
        ctx.logger.info('【handleCallback】', ctx.request.body);
        const {taskId} = ctx.request.body;
        const result = await service.task.fetchTask({taskId});
        ctx.success();
        let taskNote = {
            status: 'init',
            note: {message: 'init'},
        };
        switch (result.code) {
            case -1:
            case 106:
            case 102:
            case 204:
                taskNote = {
                    status: 'failure',
                    note: {message: result.msg},
                };
                break;
            case 1:
                if (result.data.data) {
                    result.data = {
                        data: JSON.parse(JSON.parse(result.data.data).post_data.strForNextStep),
                        loginParam: result.data.loginParam
                    }
                }
                taskNote = {
                    status: 'next',
                    note: {message: 'waiting', nextStep: result.data.loginParam},
                };
                break;
            case 110:
                taskNote = {
                    status: 'query',
                    note: {message: 'login success'},
                };
                break;
            case 0:
                taskNote = {
                    status: 'success',
                    note: {message: 'task success'},
                };
                try {
                    const order = await service.order.getOneByTaskId(taskId);
                    if (!order) {
                        throw new Error('任务已经结束了');
                    }
                    const {orderId, appkey} = order;
                    const script = await service.scripts.fetchOneScripts(result.data.cityId);
                    result.cityId = Number(result.data.cityId);
                    result.cityName = script.hub_name || 'unknown';
                    result.taskId = taskId;
                    result.data = JSON.parse(result.data.data);
                    result.orderSn = orderId;
                    result.orderId = orderId;
                    result.appKey = appkey;
                    delete result.code;
                    delete result.msg;
                    await service.storage.write(result);
                    await service.partner.notice(order);
                } catch (err) {
                    ctx.logger.error('handleCallback', err, JSON.stringify(result));
                    taskNote = {
                        status: 'failure',
                        note: {message: err.message},
                    };
                }
                break;
            default:
                ctx.logger.warn('handleCallback', JSON.stringify(result));
                break;
        }
        await service.cache.set({
            key: taskId,
            value: taskNote,
        });
    }
}

module.exports = TaskController;
