'use strict';

const Service = require('egg').Service;
const nodeExcel = require('node-xlsx');
const urlencode = require('urlencode');
const moment = require('moment');

class UserDetailService extends Service {

  async findAll(type, where, attributes, { size, page }) {
    const { ctx } = this;
    return await ctx.yizhiModel[type].findAll({
      attributes,
      where,
      offset: size * (page - 1),
      limit: size,
    });
  }

  async create(type, data) {
    const { ctx } = this;
    await ctx.yizhiModel[type].create(data);
  }

  async findOne(type, where, attributes) {
    const { ctx } = this;
    return await ctx.yizhiModel[type].findOne({
      attributes,
      where,
    });
  }

  async count(type, where = {}) {
    const { ctx } = this;
    const ret = await ctx.yizhiModel[type].findOne({
      attributes: [
        [
          ctx.yizhiModel.fn('COUNT', ctx.yizhiModel.col('id')), 'total',
        ],
      ],
      where,
    });
    return ret.get('total');
  }

  async analyseInfo({ start_date, end_date, product }) {
    const { ctx } = this;

    const user_id = ctx.userId;

    const ret = {
      product: 'all',
      overview: {
        use_money: 0,
        use_amount: 0,
      },
      list: []
    }

    let where = { user_id };
    if (product) {
      ret.product = product;
      where.service = product;
    }

    const appKeys = await ctx.yizhiModel.UserService.findAll({
      attributes: ['service', 'appkey'],
      where
    });

    if (appKeys) {
      const appList = [];
      appKeys.forEach((value, index) => {
        if (!appList.includes(value.appkey)) {
          appList.push(value.appkey);
        }
      });
      let type = 'YzbillCost';
      if (start_date.length === '7') {
        type = 'YzbillCostMonth';
      }
      where = {
        user_id: ctx.userId,
        appKey: appList,
        day: {
          $lte: end_date,
          $gte: start_date
        },
      };

      const detail = await ctx.dataModel[type].findAll({
        attributes: ['day', 'pull_volume', 'cost'],
        where,
        order: [['day', 'desc']]
      })
      const listMap = new Map();

      for (let i = 0; i < detail.length; i++) {
        if (listMap.has(detail[i].day)) {
          let obj = listMap.get(detail[i].day);
          obj.money += parseFloat(detail[i].cost);
          obj.amount += detail[i].pull_volume;
          listMap.set(detail[i].day, obj);
        } else {
          listMap.set(detail[i].day, { money: parseFloat(detail[i].cost), amount: detail[i].pull_volume });
        }
        ret.overview.use_money += Number(parseFloat(detail[i].cost).toFixed(2));
        ret.overview.use_amount += detail[i].pull_volume;
      }
      if (listMap.size > 0) {
        listMap.forEach((value, key) => {
          ret.list.push({
            date: key,
            money: Number((value.money).toFixed(2)),
            amount: Number((value.amount).toFixed(0))
          })
        })
      }
    }
    return ret;
  }

  async analyseDownload({ start_date, end_date, product }) {
    const { ctx } = this;

    const user_id = ctx.userId;
    let buffer = nodeExcel.build([{ name: 'sheet1', data: [[]] }]);

    let where = { user_id };
    if (product) {
      where.service = product;
    }

    const appKeys = await ctx.yizhiModel.UserService.findAll({
      attributes: ['service', 'appkey'],
      where
    });

    if (appKeys) {
      const appList = [];
      appKeys.forEach((value, index) => {
        if (!appList.includes(value.appkey)) {
          appList.push(value.appkey);
        }
      });
      where = {
        user_id: ctx.userId,
        appKey: appList,
        day: {
          $lte: end_date,
          $gte: start_date
        },
      };

      const detail = await ctx.dataModel.YzbillCost.findAll({
        attributes: ['appKey', 'service', 'day', 'query_volume', 'pull_volume', 'price', 'cost'],
        where,
        order: [['day', 'desc']]
      })

      let data = [['appKey', '产品名称', '日期', '查询量（请求量）', '产品调用量（结果拉取量）', '产品单价', '消耗金额']];

      for (let i = 0; i < detail.length; i++) {
        data.push([detail[i].appKey, detail[i].service, detail[i].day, detail[i].query_volume, detail[i].pull_volume, detail[i].price, detail[i].cost])
      }

      buffer = nodeExcel.build([{ name: 'sheet1', data }]);
    }
    ctx.set('Content-Type', 'application/octet-stream');
    let name = urlencode(moment(new Date()).format('YYYY-MM-DD') + '.xlsx', "utf-8");
    ctx.set("Content-Disposition", "attachment; filename* = UTF-8''" + name);
    return buffer;
  }
}

module.exports = UserDetailService;
