'use strict';


const REDIS_CACHE = Symbol('Context#RedisCache');
const NODE_CACHE = Symbol('Context#NodeCache');

const NodeCache = require('node-cache');

class Cache {
  constructor(app) {
    this.app = app;
    this.name = 'unknown-cache';
  }

  async val(key, next, ttl) {
    let data = await this.get(key);
    if (data) {
      return data;
    }
    if (next instanceof Promise) {
      data = await next;
    } else {
      data = await next(key);
    }
    await this.set(key, data, ttl);
    return data;
  }

  async get(key) {
    const startTime = +new Date();
    const ret = await this._get(key);
    let jsonText = JSON.stringify(ret);
    if (jsonText === undefined) {
      jsonText = 'undefined';
    } else if (jsonText.length >= 128) {
      if (/^\{/.test(jsonText)) {
        jsonText = '{...}';
      } else if (/^\[/.test(jsonText)) {
        jsonText = '[...]';
      } else {
        jsonText = jsonText.substr(0, 125) + '...';
      }
    }
    this.app.logger.info(`[cache](${+new Date() - startTime}ms) ${this.name}.${key}: ${jsonText}`);
    return ret;
  }

  async set(key, value, ttl = 60) {
    return await this._set(key, value, ttl);
  }

  async ttl(key) {
    return await this._ttl(key);
  }

  async lock(key, ttl, fn) {
    return await this._lock(key, ttl, fn);
  }
}

class RedisCacheWrap extends Cache {
  constructor(app, redis) {
    super(app);
    this.redis = redis;
    this.name = 'redis-cache';
    redis.defineCommand('lock', {
      numberOfKeys: 1,
      lua: 'return redis.call("set", KEYS[1], ARGV[1], "NX", "EX", ARGV[2])',
    });
  }

  async _get(key) {
    const { redis } = this;
    let value = await redis.get(key);
    if (value === null) {
      value = undefined;
    }
    if (value) {
      value = JSON.parse(value);
    }
    return value;
  }
  async _set(key, value, ttl) {
    const { redis } = this;
    value = JSON.stringify(value);
    if (ttl > 0) {
      await redis.set(key, value, 'EX', ttl);
    } else {
      await redis.set(key, value);
    }
  }
  async _ttl(key) {
    const { redis } = this;
    const r = await redis.ttl(key);
    return r < 0 ? undefined : r;
  }

  async _lock(key, ttl, fn) {
    const { app, redis } = this;
    const redisLockName = app.name + '/' + app.config.env + '/lock/' + key;
    // 获取当前时间秒和截止时间
    const currentTime = Math.round(new Date().getTime() / 1000);
    const expireTime = +ttl + currentTime;
    // 加锁lua脚本，key存在返回null，不存在则设置key-value并返回OK
    const lock = await redis.lock(redisLockName, expireTime, ttl);
    if (!lock) {
      // 如存在锁判断是否超时
      const preLockTime = +await redis.get(redisLockName);
      if (preLockTime > currentTime) {
        throw Error('系统繁忙，请稍后再试！');
      }
      // 原子方法getset，重新设置值，并返回之前设置的值
      // 如超时重新设置value，并再次判断是否有其他进程获取到锁
      const resetLockTime = +await redis.getset(redisLockName, expireTime);
      if (resetLockTime !== preLockTime) {
        throw Error('系统繁忙，请稍后重试！');// 表示有其他进程获取到锁
      }
    }
    // 执行fn逻辑并删除锁
    try {
      return await fn.apply(this, []);
    } finally {
      await redis.del(redisLockName);
    }
  }
}

class NodeCacheWrap extends Cache {
  constructor(app) {
    super(app);
    this.cache = new NodeCache();
    this.name = 'node-cache';
  }

  async _get(key) {
    return this.cache.get(key);
  }
  async _set(key, value, ttl) {
    const { cache } = this;
    value = JSON.parse(JSON.stringify(value));
    if (ttl > 0) {
      cache.set(key, value, ttl);
    } else {
      cache.set(key, value);
    }
  }
  async _ttl(key) {
    return this.cache.getTtl(key);
  }
}

module.exports = {
  get memcache() {
    if (!this[REDIS_CACHE]) {
      this[REDIS_CACHE] = new RedisCacheWrap(this, this.redis);
    }
    return this[REDIS_CACHE];
  },
  get cache() {
    if (!this[NODE_CACHE]) {
      this[NODE_CACHE] = new NodeCacheWrap(this);
    }
    return this[NODE_CACHE];
  },
};

