'use strict';

const Service = require('egg').Service;
const moment = require('moment');

class UserService extends Service {

  async getUserInfo() {
    const { ctx, config } = this;
    if (!ctx.header.authorization) {
      ctx.throw(400, 'no Authorization');
    }
    const user = await this.app.cache.get(ctx.header.authorization) || '';
    if (user) {
      return this.app.cache.get(ctx.header.authorization);
    }
    const ret = await ctx.curl(config.userInfoUrl, {
      method: 'GET',
      headers: {
        authorization: ctx.header.authorization,
      },
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchParams,${config.userInfoUrl}`, 'result:', JSON.stringify(ret.data));
    if (!(ret.status === 200 && ret.data && ret.data.id)) {
      ctx.failed('no user');
    }
    this.app.cache.set(ctx.header.authorization, ret.data, 300);
    return ret.data;
  }

  async fetchAccount(where = {}, { size, page }) {
    const { ctx } = this;
    return await ctx.yizhiModel.Account.findAll({
      attributes: ['user_id', 'account', 'name', 'company', 'dd_name', 'status'],
      where,
      offset: size * (page - 1),
      limit: size,
    });
  }

  async serviceList(where) {
    const { ctx } = this;
    return await ctx.yizhiModel.UserService.findAll({
      attributes: ['appkey', 'service'],
      where,
    });
  }

  async count(where = {}) {
    const { ctx } = this;
    const ret = await ctx.yizhiModel.Account.findOne({
      attributes: [
        [
          ctx.yizhiModel.fn('COUNT', ctx.yizhiModel.col('user_id')), 'total',
        ],
      ],
      where,
    });
    return ret.get('total');
  }

  async update(where, params, type) {
    const { ctx } = this;
    const ret = await ctx.yizhiModel[type].findOne({
      where,
    });
    if (!ret) {
      this.ctx.throw(400, '记录不存在');
    }
    await ret.update(params);
  }

  async findOne(type, where, attributes) {
    const { ctx } = this;
    return await ctx.yizhiModel[type].findOne({
      attributes,
      where,
    });
  }

  async accountInfo(user_id) {
    const { ctx } = this;
    const { account } = await this.findOne('Account', { user_id }, ['account']);
    const account_info = { account_name: account, product: [] };
    const product = await ctx.yizhiModel.UserService.findAll({
      attributes: ['service', 'appkey'],
      where: { user_id },
      distinct: true,
    });
    for (let i = 0; i < product.length; i++) {
      account_info.product.push({
        id: i,
        name: product[i].service
      });
    }
    //昨日调用信息
    const lastday_info = {
      use_amount: 0,
      use_money: 0,
      left_money: 0
    }

    const last_day = moment(new Date()).subtract(1, 'day').format('YYYY-MM-DD');
    const last_month = moment(new Date()).subtract(1, 'day').format('YYYYMM');
    const lastday_data = await ctx.dataModel.yzBillCost.findOne({
      attributes: [
        [ctx.model.fn('SUM', 'pull_volumn'), 'amount'],
        [ctx.model.fn('SUM', 'cost'), 'cost'],
      ],
      where: { user_id, day: last_day },
    })
    lastday_info.use_amount = lastday_data.amount || 0;
    lastday_info.use_money = lastday_data.cost || 0;

    const left_money = await ctx.dataModel.YzBillBalance.findOne({
      attributes: ['balance_fn'],
      where: { user_id, month: last_month },
    })
    lastday_info.left_money = left_money.balance_fn || 0;

    const detail = []
    const bill = await ctx.dataModel.YzBillBalance.findAll({
      attributes: ['month', 'company', 'balance_bg', 'recharge_m', 'cost_m', 'balance_fn'],
      where: { user_id },
      order: [['month', 'DESC']],
    })
    for (let i = 0; i < bill.length; i++) {
      detail.push({
        name:bill[i].company,
        date:bill[i].month,
        start_left:bill[i].balance_bg,
        middle_recharge:bill[i].recharge_m,
        middlefee:bill[i].cost_m,
        endfee:bill[i].balance_fn,
      })
    }

    return {
      account_info,
      lastday_info,
      detail
    }
  }


}
module.exports = UserService;
