'use strict';

const Controller = require('egg').Controller;

class OrderController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      appKey: 'string',
      userId: {
        type: 'string',
        required: false,
      },
      orderId: {
        type: 'string',
        required: false,
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: '订单号格式错误',
      },
      phone: {
        type: 'string',
        format: /^(((13[0-9]{1})|(14[0-9]{1})|(15[4-9]{1})|(16[0-9]{1})|(17[0-9]{1})|(18[0-9]{1})|(19[0-9]{1}))+\d{8})$/,
        message: 'phone格式错误',
        required: true,
      },
    };

    this.showRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          orderId: {
            type: 'string',
            required: false,
          },
          orderSn: {
            type: 'string',
            required: false,
          },
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };

    this.fetchRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          appKey: 'string',
          timestamp: 'string',
          token: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { appKey, userId, notifyUrl, backUrl, orderId, phone } = ctx.request.body;
    // const appKeyInfo = await service.partner.fetchInfo(appKey);
    // if (!(appKeyInfo && appKeyInfo.valid === true && appKeyInfo.enableYys === true)) {
    //   return ctx.fail('未开通此服务');
    // }
    const areaInfo = await service.task.phoneArea(phone);
    const object = {
      orderId,
      userId: userId || '',
      notifyUrl: notifyUrl || (ctx.app.notifyMap && ctx.app.notifyMap.get(appKey) && ctx.app.notifyMap.get(appKey).notifyUrl) || '',
      backUrl: backUrl || '',
      appKey,
      status: 'init',
      phone,
      city: areaInfo.province,
      operator: areaInfo.operator,
    };
    if (orderId) {
      await service.order.update(object);
    } else {
      object.orderId = ctx.helper.getUuid();
      await service.order.create(object);
    }
    ctx.success({ orderId: object.orderId });
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.showRule);
    const { orderId } = ctx.params;
    await service.signature.signatureCheck(ctx.request.body);
    const data = await service.order.getOneByOrderId(orderId);
    if (data) {
      if (data.status === 'success') {
        const result = await service.storage.read(orderId);
        ctx.success(result);
        return;
      }
      ctx.success({
        status: data.status,
      });
      return;
    }
    ctx.fail('无此订单号');
  }

  async partnerShow() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.showRule);
      let { appKey, orderId, orderSn } = ctx.request.body.params;
      orderId = orderId || orderSn;
      await service.signature.signatureCheck(ctx.request.body);
      const appKeyInfo = await service.partner.fetchInfo(appKey);
      if (!(appKeyInfo && appKeyInfo.valid === true && appKeyInfo.enableHfData === true)) {
        return ctx.fail('未开通此服务');
      }
      const data = await service.order.getOneByOrderId(orderId);
      if (data) {
        if (data.status === 'success') {
          const result = await service.storage.read(orderId, appKey);
          ctx.body = {
            code: 0,
            data: result,
          };
          return;
        }
      }
      ctx.body = {
        code: -1,
        msg: '无此订单号',
      };
    } catch (err) {
      ctx.logger.error('partnerShow', JSON.stringify(err));
      ctx.error(err);
    }
  }

  async orderShow() {
    const { ctx, service } = this;
    const { appKey, orderId } = ctx.params;
    const appKeyInfo = await service.partner.fetchInfo(appKey);
    if (!(appKeyInfo && appKeyInfo.valid === true && appKeyInfo.enableHfView === true)) {
      return ctx.fail('未开通此服务');
    }
    const data = await service.order.getOneByOrderId(orderId);
    if (data) {
      if (data.status === 'success' && (data.appKey === appKey)) {
        const result = await service.storage.read(orderId, appKey);
        const ret = service.washData.dealData(result);
        ctx.success(ret);
        return;
      }
    }
    ctx.fail('无此订单号');
  }

  async fetchOrderId() {
    const { ctx, service } = this;
    try {
      ctx.validate(this.fetchRule);
      const { appKey } = ctx.request.body.params;
      const orderId = await service.signature.createOrderId(ctx.request.body);
      await service.order.create({
        orderId,
        notifyUrl: ctx.app.notifyMap.get(appKey) && ctx.app.notifyMap.get(appKey).notifyUrl || '',
        appKey,
        status: 'init',
      });
      ctx.body = {
        code: 0,
        data: {
          orderSn: orderId,
        },
        msg: '',
      };
    } catch (err) {
      ctx.logger.error('fetchOrderId', JSON.stringify(err));
      ctx.error(err);
    }
  }
}

module.exports = OrderController;
