'use strict';

const Service = require('egg').Service;

class TaskService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config: { taskAPI, LOCK_KEY, phoneAreaAPI } } = this;
    this.taskAPI = taskAPI;
    this.phoneAreaAPI = phoneAreaAPI;
    this.baseURL = taskAPI.host;
    this.citylistURL = taskAPI.citylist;
    this.LOCK_KEY = LOCK_KEY;
  }


  _request(url, opts) {
    const { ctx, baseURL } = this;
    url = `${baseURL}${url}`;
    opts = {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      ...opts,
    };
    return ctx.curl(url, opts);
  }

  _checkSuccess(result) {
    if (result.status !== 200) {
      const errorMsg = result.data && result.data.error_msg ? result.data.error_msg : 'unknown error';
      this.ctx.throw(result.status, errorMsg);
    }
    if (result.data.code !== 0) {
      this.ctx.throw(400, { message: result.data.msg || result.data.data.img, code: result.data.code || -1 });
    }
  }

  async configs() {
    const { taskAPI, ctx } = this;
    const { configUrl } = taskAPI;
    const result = await this._request(configUrl, {
      method: 'get',
      contentType: 'json',
    });
    ctx.logger.info(`【Task】config ${configUrl} params`, JSON.stringify(result.data));
    this._checkSuccess(result);
    return result;
  }

  async create(data) {
    const { ctx, taskAPI } = this;
    const { createTaskUrl } = taskAPI;
    const result = await this._request(createTaskUrl, {
      method: 'post',
      data: {
        cityId: data.cityId,
        orderId: data.orderId,
        source: 'yysH5new',
      },
      contentType: 'json',
    });
    ctx.logger.info(`【Task】create ${createTaskUrl} cityId: ${JSON.stringify(data)} result:`, JSON.stringify(result.data.code));
    this._checkSuccess(result);
    return result.data.data;
  }

  async fetchCapture({ taskId, type, data }) {
    const { taskAPI, ctx } = this;
    const { fetchCodeUrl } = taskAPI;
    const result = await this._request(fetchCodeUrl, {
      method: 'post',
      data: {
        taskId,
        type,
        data,
      },
      contentType: 'json',
    });
    ctx.logger.info(`【Task】fetchCapture ${fetchCodeUrl} params`, JSON.stringify({
      taskId,
      type,
    }), JSON.stringify(result.data));
    this._checkSuccess(result);
    return result.data.data;
  }

  async submit(data) {
    const { taskAPI, ctx } = this;
    const { submitTaskUrl } = taskAPI;
    const result = await this._request(submitTaskUrl, {
      method: 'post',
      data,
      contentType: 'json',
    });
    ctx.logger.info(`【Task】submit ${submitTaskUrl} params`, JSON.stringify(data), JSON.stringify(result.data));
    this._checkSuccess(result);
    return result;
  }

  async fetchTask(data) {
    const { taskAPI, ctx } = this;
    const { fetchTaskUrl } = taskAPI;
    const result = await this._request(fetchTaskUrl, {
      method: 'post',
      data,
      contentType: 'json',
    });
    ctx.logger.info(`【Task】fetchTask ${fetchTaskUrl} params`, JSON.stringify(data), JSON.stringify(result.data));
    return result.data;
  }

  async phoneArea(phone) {
    const { ctx, phoneAreaAPI } = this;
    const { alhost } = phoneAreaAPI;
    const phoneInfo = {
      phone,
      province: '',
      operator: '',
    };
    const result = await ctx.curl(alhost + phone, {
      method: 'get',
      timeout: [3000, 3000],
    });
    if (result.status !== 200) {
      ctx.throw('400', '请检查手机号是否有效');
    }
    const html = ctx.helper.decode(result.data);
    if (!html.match(/province:'(.*?)'/) || !html.match(/catName:'(.*?)'/)) {
      ctx.throw('400', '请检查手机号是否有效');
    }
    phoneInfo.province = html.match(/province:'(.*?)'/)[1];
    phoneInfo.operator = html.match(/catName:'(.*?)'/)[1];
    ctx.logger.info('phoneArea', alhost + phone, JSON.stringify(phoneInfo), html, phone);
    if (phoneInfo.province === '香港') {
      phoneInfo.province = '广东';
    }
    if (phoneInfo.operator.match(/移动|联通|电信/)) {
      phoneInfo.operator = phoneInfo.operator.match(/移动|联通|电信/)[0];
    } else {
      phoneInfo.operator = '虚拟运营商';
    }
    ctx.logger.info('phoneArea result', JSON.stringify(phoneInfo), phone);
    return phoneInfo;
  }

}

module.exports = TaskService;
