'use strict';

const Controller = require('egg').Controller;

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      orderId: {
        type: 'string',
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: 'orderId格式错误',
        required: true,
      },
      scriptId: {
        type: 'number',
        required: true,
      },
    };

    this.submitRule = {
      phone: { type: 'string', format: /^1\d{10}$/, required: false },
      ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false },
    };

    this.fetchCaptureRule = {
      type: { type: 'enum', values: ['phone_code'] },
    };

    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
    };

    this.cityListRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async create() {
    const { ctx, service } = this;
    ctx.validate(this.createRule);
    const { scriptId, orderId } = ctx.request.body;
    const taskId = await service.task.create(scriptId);
    await service.order.update({ orderId, cityId: scriptId, taskId });
    await service.cache.set({
      key: taskId,
      value: { status: 'init', note: { message: 'init' } },
    });
    ctx.success({ taskId });
  }

  async submit() {
    const { ctx, service, config } = this;
    ctx.validate(this.submitRule);
    const { taskId } = ctx.params;
    await service.cache.set({
      key: taskId,
      value: { status: 'login', note: { message: 'login' } },
    });
    await service.task.submit({
      taskId,
      data: ctx.request.body,
      callbackUrl: config.callbackUrl,
    });
    ctx.success({ taskId });
  }

  async fetchCapture() {
    const { ctx, service } = this;
    ctx.validate(this.fetchCaptureRule);
    const { taskId } = ctx.params;
    const { type, data } = ctx.request.body;
    const result = await service.task.fetchCapture({
      taskId,
      type,
      data,
    });
    ctx.success(result);
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.taskIdRule, ctx.params);
    const { taskId } = ctx.params;
    const data = await service.cache.get({ key: taskId });
    ctx.success(data);
  }

  async handleCallback() {
    const { ctx, service } = this;
    const { taskId } = ctx.request.body;
    ctx.body = {
      code: 200,
    };
    const result = await service.task.fetchTask({ taskId });
    let taskNote = {
      status: 'init',
      note: { message: 'init' },
    };
    switch (result.code) {
    case -1:
    case 106:
    case 102:
    case 204:
      taskNote = {
        status: 'failure',
        note: { message: result.msg },
      };
      break;
    case 1:
      if (result.data.data) {
        result.data = {
          data: JSON.parse(JSON.parse(result.data.data).post_data.strForNextStep),
          loginParam: result.data.loginParam,
        };
      }
      taskNote = {
        status: 'next',
        note: { message: 'waiting', nextStep: result.data },
      };
      break;
    case 110:
      taskNote = {
        status: 'query',
        note: { message: 'login success' },
      };
      break;
    case 0:
      taskNote = {
        status: 'wash',
        note: { message: 'wash' },
      };
      try {
        const order = await service.order.getOneByTaskId(taskId);
        if (!order) {
          throw new Error('任务不存在');
        }
        const { orderId, netMark } = order;
        await service.washData.wash(result, orderId, netMark);
        await service.cache.set({
          key: taskId,
          value: taskNote,
        });
        return;
      } catch (err) {
        ctx.logger.error('handleCallback', JSON.stringify(err), JSON.stringify(result));
        taskNote = {
          status: 'failure',
          note: { message: err.message },
        };
      }
      break;
    default:
      ctx.logger.warn('handleCallback', JSON.stringify(result));
      break;
    }
    await service.cache.set({
      key: taskId,
      value: taskNote,
    });
  }

  async dataCallback() {
    const { ctx, service } = this;
    ctx.body = {
      code: 200,
    };
    const { taskId, state } = ctx.request.body;
    const order = await service.order.getOneByTaskId(taskId);
    if (!order) {
      throw new Error('任务不存在');
    }
    let taskNote = {
      status: 'failure',
      note: { message: '存储数据出错' },
    };
    if (state === 'success') {
      taskNote = {
        status: 'success',
        note: { message: 'task success' },
      };
    }
    await service.partner.notice(order, state);
    await service.cache.set({
      key: taskId,
      value: taskNote,
    });
  }

}

module.exports = TaskController;
