'use strict';

const Service = require('egg').Service;

class ScriptsService extends Service {
  constructor(ctx) {
    super(ctx);
    const { config } = this;
    const { scriptsAPI } = config;
    this.scriptsAPI = scriptsAPI;
    this.baseURL = scriptsAPI.host;
    this.fetchScriptsUrl = scriptsAPI.fetchScriptsUrl;
    this.fetchOneScriptUrl = scriptsAPI.fetchOneScriptUrl;
    this.fetchParamsInfoUrl = scriptsAPI.fetchParamsInfoUrl;
    this.fetchHelpUrl = scriptsAPI.fetchHelpUrl;
    this.redisScriptsKey = scriptsAPI.redisScriptsKey;
    this.redisParamsKey = scriptsAPI.redisParamsKey;
    this.redisScriptListKey = scriptsAPI.redisScriptListKey;
    this.cityListUrl = scriptsAPI.cityListUrl;
  }

  async fetchScripts() {
    const { baseURL, fetchScriptsUrl, redisScriptsKey, ctx } = this;
    const data = await this.app.redis.get(redisScriptsKey);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Scripts】fetchScriptsFromRedis', 'result:', data, err);
        await this.app.redis.del(redisScriptsKey);
      }
    }
    const result = await ctx.curl(baseURL + fetchScriptsUrl, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchScripts,${baseURL + fetchScriptsUrl}`, 'result:', JSON.stringify(result.data));
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisScriptsKey, JSON.stringify(result.data), 'EX', 300);
    }

    return result.data;
  }
  async fetchOneScripts(scriptId) {
    const { baseURL, fetchOneScriptUrl, redisScriptListKey, ctx } = this;
    const data = await this.app.redis.get(redisScriptListKey + scriptId);
    if (data) {
      try {
        return JSON.parse(data);
      } catch (err) {
        ctx.logger.error('【Scripts】fetchOneScriptsFromRedis', 'result:', data, err);
        await this.app.redis.del(redisScriptListKey + scriptId);
      }
    }
    const result = await ctx.curl(baseURL + fetchOneScriptUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    if (result.data && (String(result.data.id) === scriptId)) {
      await this.app.redis.set(redisScriptListKey + scriptId, JSON.stringify(result.data), 'EX', 60);
    }
    return result.data;
  }

  async fetchParams(refresh = false) {
    const { baseURL, fetchParamsInfoUrl, redisParamsKey, ctx } = this;
    if (!refresh) {
      const data = await this.app.redis.get(redisParamsKey);
      if (data) {
        try {
          return JSON.parse(data);
        } catch (err) {
          ctx.logger.error('【Scripts】fetchParams', 'result:', data, err);
          await this.app.redis.del(redisParamsKey);
        }
      }
    }

    const result = await ctx.curl(baseURL + fetchParamsInfoUrl, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`【Scripts】fetchParams,${baseURL + fetchParamsInfoUrl}`, 'result:', JSON.stringify(result.data));
    if (result.data && result.data.length > 0) {
      await this.app.redis.set(redisParamsKey, JSON.stringify(result.data), 'EX', 300);
    }
    return result.data;
  }

  async fetchHelp(scriptId) {
    const { baseURL, scriptsAPI: { fetchHelpUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchHelpUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchQueryButton(scriptId) {
    const { baseURL, scriptsAPI: { fetchQueryButtonUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchQueryButtonUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchNotice(scriptId) {
    const { baseURL, scriptsAPI: { fetchNoticeUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchNoticeUrl + '/' + scriptId, {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchHubSeripts(hubId) {
    const { baseURL, scriptsAPI: { fetchHubSeriptsUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchHubSeriptsUrl + '/' + hubId + '/scripts', {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    return result.data;
  }

  async fetchScriptByCityName(name) {
    const { baseURL, scriptsAPI: { fetchScriptByCityNameUrl }, ctx } = this;
    const result = await ctx.curl(baseURL + fetchScriptByCityNameUrl + '?name=' + encodeURIComponent(name) + '&type=query', {
      charset: 'utf-8',
      timeout: [30000, 30000],
      dataType: 'json',
      contentType: 'json',
    });
    ctx.logger.info(`fetchScriptByCityName,${baseURL + fetchScriptByCityNameUrl + '?name=' + name}`, 'result:', JSON.stringify(result.data));
    if (result.status === 200) {
      return result.data.scriptId;
    }
    ctx.throw(400, '手机号码区域验证失败');
  }

  async fetchScriptName(scriptId) {
    try {
      const city = await this.fetchOneScripts(scriptId);
      return city.name || '未知';
    } catch (err) {
      return '未知';
    }
  }

  // eslint-disable-next-line no-empty-function
  async getCityList(params) {
    try {
      const { cityListUrl, ctx } = this;
      const result = await ctx.curl(cityListUrl, {
        charset: 'utf-8',
        timeout: [30000, 30000],
        dataType: 'json',
        contentType: 'json',
        method: 'POST',
        data: params,
      });
      return result.data;
    } catch (err) {
      this.ctx.logger.error(err);
      return { code: -1, msg: '系统错误,请稍后重试' };
    }
  }
}

module.exports = ScriptsService;
