'use strict';

const Controller = require('egg').Controller;
const md5 = require('md5-node');

class TaskController extends Controller {
  constructor(ctx) {
    super(ctx);
    this.createRule = {
      orderId: {
        type: 'string',
        format: /^(\w{8})-(\w{4})-(\w{4})-(\w{4})-(\w{12})$/,
        message: 'orderId格式错误',
        required: true,
      },
      scriptId: {
        type: 'string',
        required: true,
      },
    };

    this.submitRule = {
      phone: { type: 'string', format: /^1\d{10}$/, required: false },
      ID: { type: 'string', format: /^\d{14}(\d{3})?[\dXx]$/, required: false },
    };

    this.taskIdRule = {
      taskId: { type: 'string', format: /\d+/ },
    };

    this.cityListRule = {
      sign: 'string',
      params: {
        type: 'object',
        rule: {
          token: 'string',
          appKey: 'string',
          timestamp: 'string',
        },
      },
    };
  }

  async configs() {
    const { ctx, service } = this;
    const data = await service.task.configs();
    ctx.success(data);
  }

  async create() {
    const { ctx, app, service } = this;
    ctx.validate(this.createRule);
    const { scriptId, orderId, appkey, phone } = ctx.request.body;
    let phoneData = await app.redis.get(phone);
    if (phoneData) {
      phoneData = JSON.parse(phoneData);
    }
    const tasks = await service.task.create({ cityId: scriptId, orderId });
    service.order.create({
      orderId,
      phone,
      appkey,
      taskId: tasks.taskId,
      city: phoneData.city || '',
      operator: phoneData.operator || '',
      status: 'init',
    });
    // await service.cache.set({
    //   key: String(tasks.taskId),
    //   value: { status: 'init', note: { message: 'init' } },
    // });
    ctx.success({ task_id: tasks.taskId });
  }

  async submit() {
    const { ctx, service, config } = this;
    ctx.validate(this.submitRule);
    const { args, task_id, orderId } = ctx.request.body;
    await service.task.submit({
      taskId: task_id,
      data: args,
      callbackUrl: config.callbackUrl,
    });
    service.order.update({
      status: 'login',
      params: {
        ID: md5(args.ID) || '',
        name: md5(args.name) || '',
        password: md5(args.password) || '',
      },
    }, { orderId, taskId: task_id });
    // service.cache.set({
    //   key: String(task_id) + data.data.orderId,
    //   value: { status: 'login', note: { message: 'login' } },
    // });
    ctx.success({ errcode: 0, taskId: task_id });
  }

  async fetchCapture() {
    const { ctx, service } = this;
    const { args, task_id } = ctx.request.body;
    const result = await service.task.fetchCapture({
      taskId: task_id,
      data: args,
    });
    ctx.success(result);
  }

  async fields() {
    const { ctx, service } = this;
    const { cid } = ctx.request.body;
    const data = ctx.app.cityMap.get(cid) || {};
    let formHtml = await service.index.getCityHtml(data.queryStepParam[0].loginParam, cid, { name: '', ID: '', password: ''});

    if (data.state !== 1) {
      formHtml = await service.index.getCityHtml(data.queryStepParam[0].loginParam, cid, { name: '', ID: '', password: ''});
    }
    ctx.success(formHtml);
  }

  async show() {
    const { ctx, service } = this;
    ctx.validate(this.taskIdRule, ctx.params);
    const { taskId } = ctx.params;
    const data = await service.cache.get({ key: String(taskId) });
    ctx.success(data);
  }

  async handleCallback() {
    const { ctx, service } = this;
    const { orderId, taskId } = ctx.request.body;
    ctx.body = {
      code: 200,
    };
    const result = await service.task.fetchTask({ taskId });
    // let taskNote = {
    //   status: 'init',
    //   note: { message: 'init' },
    // };
    switch (result.code) {
    case -1:
      service.order.update({
        status: 'fail',
        note: JSON.stringify(result),
      }, { orderId, taskId });
      ctx.success({ errcode: 1, data: { state: -1, result: '服务器出错，请稍后重试' } });
      // taskNote = {
      //   status: 'failure',
      //   note: { message: result.msg },
      // };
      break;
    case 1:
      // if (result.data.data) {
      //   result.data = {
      //     data: JSON.parse(JSON.parse(result.data.data).post_data.strForNextStep),
      //     loginParam: result.data.loginParam,
      //   };
      // }
      service.order.update({
        status: 'next',
        note: JSON.stringify(result),
      }, { orderId, taskId });
      ctx.success({ errcode: 0, data: { state: 1, html: service.index.getSMSHTML() } });
      // taskNote = {
      //   status: 'next',
      //   note: { message: 'waiting', nextStep: result.data },
      // };
      break;
    case 110:
      service.order.update({
        status: 'query',
        note: JSON.stringify(result),
      }, { orderId, taskId });
      ctx.success({ errcode: 0, data: { state: 5, result: '更改为Query' } });
      // taskNote = {
      //   status: 'query',
      //   note: { message: 'submit success' },
      // };
      break;
    case 0:
      // taskNote = {
      //   status: 'wash',
      //   note: { message: 'wash' },
      // };
      try {
        service.order.update({
          status: 'wash',
          note: JSON.stringify(result),
        }, { orderId, taskId });
        await service.washData.wash(result, orderId);
        await service.washData.newWash(result, orderId);
        // await service.cache.set({
        //   key: String(taskId),
        //   value: taskNote,
        // });
        ctx.success({ errcode: 2, data: { state: 0, result: '发送数据处理' } });
      } catch (err) {
        ctx.logger.error('handleCallback 0', JSON.stringify(err), JSON.stringify(result));
        service.order.update({
          status: 'queryFail',
          note: '运营商认证分析失败',
        }, { orderId, taskId });
        ctx.success({ errcode: 1, data: { state: -1, result: '运营商认证分析失败' } });
        // taskNote = {
        //   status: 'failure',
        //   note: { message: err.message },
        // };
      }
      break;
    default:
      ctx.logger.warn('handleCallback', JSON.stringify(result));
      service.order.update({
        status: 'unknownFail',
        note: JSON.stringify({ code: result.code, msg: result.msg }),
      }, { orderId, taskId });
      ctx.success({ errcode: 2, data: { state: 0, result: '无关紧要的数据' } });
      break;
    }
    // await service.cache.set({
    //   key: String(taskId),
    //   value: taskNote,
    // });
  }

  async dataCallback() {
    const { ctx, service } = this;
    ctx.body = {
      code: 200,
    };
    const { taskId, state } = ctx.request.body;
    // const order = await service.order.getOneByTaskId(String(taskId));
    // if (!order) {
    //   throw new Error('任务不存在');
    // }
    // let taskNote = {
    //   status: 'failure',
    //   note: { message: '存储数据出错' },
    // };
    // if (state === 'success') {
    //   taskNote = {
    //     status: 'success',
    //     note: { message: 'task success' },
    //   };
    // }
    service.order.update({
      status: 'unknownFail',
      note: JSON.stringify({ code: result.code, msg: result.msg }),
    }, { orderId, taskId });
    await service.partner.notice(order, state);
    // await service.cache.set({
    //   key: String(taskId),
    //   value: taskNote,
    // });
  }

  async newdataCallback() {
    const { ctx, service } = this;
    ctx.body = {
      code: 200,
    };
    const { taskId, state } = ctx.request.body;
    const order = await service.order.getOneByTaskId(String(taskId));
    if (!order) {
      throw new Error('任务不存在');
    }
    let taskNote = {
      status: 'failure',
      note: { message: '存储数据出错' },
    };
    if (state === 'success') {
      taskNote = {
        status: 'success',
        note: { message: 'task success' },
      };
    }
    await service.partner.notice(order, state, 1);
    await service.cache.set({
      key: String(taskId),
      value: taskNote,
    });
  }
}

module.exports = TaskController;
