
'use strict';

const Service = require('egg').Service;
const R = require('ramda');
const moment = require('moment');
class InstitutionService extends Service {


    //机构列表页
    async getInstitutions(input) {

        const { ctx } = this;
        const { cat, age, institution } = input;
        let where = { status: 1, is_deleted: 0 };
        if (cat) {
            const cat_ret = await ctx.prometheusModel.CourseCat.one({ where: { id: cat } });
            const cat_id = cat_ret.id;
            const cat_level = cat_ret.level;
            const next_level = cat_level + 1;
            const next_next_level = cat_level + 2;
            const next_cat = await ctx.prometheusModel.CourseCat.all({ where: { level: next_level, parent_id: cat_id } });
            const next_next_cat = await ctx.prometheusModel.CourseCat.all({ where: { level: next_next_level, first_id: cat_id } });
            let cat_ids = [cat_id,];
            cat_ids = cat_ids.concat(R.pluck('id', next_cat)).concat(R.pluck('id', next_next_cat));
            const institutions = await ctx.prometheusModel.CourseInstitutionToCat.all({ where: { cat_id: { $in: cat_ids } } });
            where.id = { $in: R.pluck('institution_id', institutions) };
        }
        if (age) {
            where.max_age = { $gte: age };
            where.min_age = { $lte: age };
        }
        if (institution) {
            where.corner = { $ne: '' };
        }
        const include = [{ model: ctx.prometheusModel.CourseArea, where: { status: 1, is_deleted: 0 }, attributes: ['id', 'institution_id', 'name', 'address', 'lat', 'lng'] }];
        const attributes = ['id', 'name', 'type', 'image', 'establishment_time', 'class_type', 'teacher_count', 'teacher_experience', 'corner', 'min_age', 'max_age', 'price', 'characteristic',];
        const institutions = await ctx.prometheusModel.CourseInstitution.findAll({ attributes, include, where });

        const institution_area_list = await this.getInstitutionAreaList(institutions);
        const area_lbs = await this.computeDistance(institution_area_list);
        const institution_areas = await this.findShortestDistanceAreas(institution_area_list, area_lbs);
        const ret = await this.formatInstitutions(institution_areas);
        return ret;
    }

    /**
     * 机构详情页
     */
    async getInstitution({ institution_id, area_id }) {

        const { ctx } = this;
        const institution = await ctx.prometheusModel.CourseInstitution.one({ where: { id: institution_id } });
        const teachers = await this.getTeachers({ institution_id, limit: 6 });
        const classes = await this.getClasses({ institution_id, limit: 4 });
        const areas = await this.getInstitutionAreas({ institution_id, limit: 1000 });
        const current_area = await ctx.prometheusModel.CourseArea.one({ id: area_id });
        let institution_detail = await this.formatInstitutions([institution]);
        institution_detail = institution_detail[0];
        institution_detail.address = current_area.address;
        institution_detail.phone = current_area.phone;
        institution_detail.photo_album = institution.image.split(';');

        return { institution_detail, teachers: teachers.rows, classes: classes.rows, areas: areas.rows };
    }

    async getTeacher(teacher_id) {

        const { ctx } = this;
        const where = { id: teacher_id };
        let teacher = await ctx.prometheusModel.CourseTeacher.one({ where });
        teacher.dataValues.point_tags = teacher.point.split(';');
        teacher.dataValues.work_experience_tags = teacher.work_experience.split(';');

        return teacher;
    }

    async getClass(class_id) {

        const { ctx } = this;
        const where = { id: class_id };
        let classs = await ctx.prometheusModel.CourseClass.one({ where });
        classs.dataValues.age_text = `${classs.min_age}-${classs.max_age}岁`;
        classs.dataValues.point_tags = classs.point.split(';');
        classs.dataValues.photo_album = classs.image.split(';');
        return classs;
    }

    async getTeachers(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'avatar', 'teacher_experience', 'lesson', 'educational_background', 'certificate'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const teachers = await ctx.prometheusModel.CourseTeacher.list({ attributes, page, limit, where });

        return teachers;

    }

    async getClasses(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'image', 'type', 'price'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const classes = await ctx.prometheusModel.CourseClass.list({ attributes, page, limit, where });
        let ret = [];
        for (let i in classes.rows) {
            let classs = classes.rows[i];
            const photo_album = institution.image.split(';');
            classs.dataValues.image = photo_album[0];
            classs.dataValues.price_text = classs.price ? classs.price : '现场咨询';
            ret.push(classs);
        }

        return { page: classes.page, count: classes.count, rows: ret };

    }

    async getInstitutionAreas(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'address', 'phone'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const areas = await ctx.prometheusModel.CourseArea.list({ attributes, page, limit, where });

        return areas;

    }

    async getInstitutionAreaList(institutions) {

        const { ctx } = this;

        const ret = [];
        for (let i in institutions) {
            let institution = institutions[i];
            if (!institution.course_areas || institution.course_areas.length === 0) continue;
            const course_areas = institution.course_areas;
            for (let j in course_areas) {
                const course_area = course_areas[j];
                ret.push({
                    id: institution.id,
                    name: institution.name,
                    type: institution.type,
                    image: institution.image,
                    establishment_time: institution.establishment_time,
                    class_type: institution.class_type,
                    teacher_count: institution.teacher_count,
                    teacher_experience: institution.teacher_experience,
                    corner: institution.corner,
                    min_age: institution.min_age,
                    max_age: institution.max_age,
                    price: institution.price,
                    characteristic: institution.characteristic,
                    area_id: course_area.id,
                    area_name: course_area.name,
                    area_address: course_area.address,
                    lat: course_area.lat,
                    lng: course_area.lng,
                });
            }
        }

        return ret;
    }

    //distance=3km
    async computeDistance(lbs_array, distance = 3) {

        const { ctx } = this;
        const from = { lat: '30.291210', lng: '120.068642' };

        const driving_results = await ctx.service.course.lbs.getLBSDistance('driving', from, lbs_array);
        const walking_results = await ctx.service.course.lbs.getLBSDistance('walking', from, lbs_array);

        const drivings = driving_results.results.map(item => {
            const distance = (item.distance / 1000).toFixed(1);
            const duration = Math.ceil(item.duration / 60);
            return { distance: distance, duration: duration }
        });
        const walkings = walking_results.results.map(item => {
            const distance = (item.distance / 1000).toFixed(2);
            const duration = Math.ceil(item.distance / 80);
            return { distance: distance, duration: duration }
        });

        const results = [];
        for (let i in lbs_array) {
            let lbs = lbs_array[i];
            const driving = drivings[i];
            const walking = walkings[i];
            const lbs_distance = driving.distance > distance ? driving : walking;
            const travel_method = driving.distance > distance ? 'driving' : 'walking';
            results.push({
                lat: lbs.lat,
                lng: lbs.lng,
                distance: lbs_distance.distance,
                duration: lbs_distance.duration,
                travel_method,
            });
        }

        return results;

    }

    async findShortestDistanceAreas(institution_areas, areas_list) {

        const { ctx } = this;

        const ret = [];
        const institution_lbs = await this.computeDistance(areas_list);
        for (let i in areas_list) {
            let area = areas_list[i];
            let institution_area = institution_areas[i];
            const lbs = institution_lbs[i];
            const area_name = institution_area.area_name;
            const distance = lbs.distance;
            const duration = lbs.duration;
            let travel_tips = `${area_name},距我${distance}km,开车${duration}分钟`;
            if (lbs.travel_method === 'walking') {
                travel_tips = `${area_name},距我${distance}km,步行${duration}分钟`;
            }
            // area.distance = distance;
            // area.duration = duration;
            // area.travel_tips = travel_tips;

            institution_area.distance = distance;
            institution_area.duration = duration;
            institution_area.travel_tips = travel_tips;

            ret.push(institution_area);
        }

        return ret;
    }

    async formatInstitutions(institutions) {

        if (!Array.isArray(institutions) || institutions.length === 0) {
            return [];
        }
        const ret = [];
        for (let i in institutions) {
            const institution = institutions[i];
            // const course_areas = institution.course_areas;
            // if (!course_areas || course_areas.length === 0) continue;
            const age_tag = institution.min_age + '-' + institution.max_age + '岁';
            const build_time = moment().format('YYYY') - institution.establishment_time;
            const tags = [age_tag, institution.class_type, '成立' + build_time + '年'];
            const photo_album = institution.image.split(';');

            ret.push({
                id: institution.id,
                name: institution.name,
                image: photo_album[0],
                establishment_time: institution.establishment_time,
                class_type: institution.class_type,
                teacher_count: institution.teacher_count,
                teacher_experience: institution.teacher_experience,
                corner: institution.corner,
                min_age: institution.min_age,
                max_age: institution.max_age,
                price: institution.price,
                tags,
                area_id: institution.area_id,//校区id TODO
                travel_tips: institution.travel_tips, //TODO
                characteristic: institution.characteristic,
            });
        }

        return ret;
    }







}

module.exports = InstitutionService;
