
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const _ = require('lodash');

class LotteryPubliciseService extends Service {

    /**
     * 根据用户的摇号楼盘和公证编号查询摇号结果
     * @param {object} {name,no,phone} 摇号楼盘 编号 用户手机号
     */
    async getLotteryPublicise(inputParams) {
        const { ctx } = this;
        //对用户的每次查询进行记录
        let data = {
            house_name: inputParams.name,
            notarization_no: inputParams.no,
            phone: inputParams.phone
        }
        let add = await ctx.realestateModel.CdLotteryUserQuery.create(data);

        //查询楼盘
        let filter = {
            where: {
                house_name: inputParams.name,
                notarization_no: inputParams.no,
            },
            order: [['creates_at', 'desc']]
        }
        let lotteryPublicise = await ctx.realestateModel.CdLotteryPublicise.findOne(filter);
        let ret = {
            status: 0,
            data: {
                times: '',
                sort: '',
                no: '',
                house_name: '',
            }
        }
        if (lotteryPublicise) {
            ret.status = 1;
            ret.data = {
                times: lotteryPublicise.times,
                sort: lotteryPublicise.select_order,
                no: lotteryPublicise.notarization_no,
                house_name: lotteryPublicise.house_name,
            }
            //如果能查询到则增加查询到的id到用户查询记录表中,方便用户查看历史的时候查询
            await ctx.realestateModel.CdLotteryUserQuery.update({ publicise_id: lotteryPublicise.id }, { where: { id: add.id } });
        }
        return ret;
    }


    /**
     * 根据用户输入的摇号楼盘的名字模糊匹配所有的楼盘
     * @param {*} houseName 
     */
    async getHouseName(houseName) {
        const { ctx } = this;
        let likeHouseName = '%' + houseName.split('').join('%') + '%';
        let filter = {
            attributes: ['DISTINCT' `house_name`],
            where: {
                house_name: { $like: likeHouseName }
            }
        }
        let houseNames = await ctx.realestateModel.CdLotteryPublicise.findAll(filter);
        let ret = { list: [] }
        ret.list = _.map(houseNames, 'house_name');
        return ret;
    }

    /**
     * 获取用户的查询记录
     */
    async getUserQueryList() {
        const { ctx } = this;
        if (!ctx.appUserId || !ctx.userId) {
            //如果没有登录就不做处理
            return false;
        }
        let query = 'SELECT aa.id query_id,aa.notarization_no no FROM (select id,notarization_no,house_name,created_at FROM cd_lottery_user_query WHERE user_id = ? AND valid = 1 ORDER BY created_at DESC) aa GROUP BY aa.house_name DESC ORDER BY aa.created_at DESC;';
        let queryFilter = {
            replacements: [ctx.userId],
            type: ctx.actModel.QueryTypes.SELECT,
        }
        let ret = await ctx.realestateModel.CdLotteryUserQuery.query(query, queryFilter);
        return ret;
    }

    /**
     * 根据用户查询记录id查询用户以前的查询公示记录
     * @param {integer} id 用户查询记录的id
     */
    async getPubliciseByQueryId(id) {
        const { ctx } = this;
        let ret = {
            status: 0,
            data: {
                times: '',
                sort: '',
                no: '',
                house_name: '',
            }
        }
        let queryFilter = {
            attributes: ['house_name', 'notarization_no', 'phone', 'publicise_id'],
            where: {
                id: id
            }
        }
        let queryInfo = await ctx.realestateModel.CdLotteryUserQuery.findOne(queryFilter);
        if (!queryInfo) {
            return ret;
        }
        if (queryInfo.publicise_id === 0) {
            //如果用户以前没有查到,此处需不需要重新搜索还是说知己网显示以前的结果
            return ret;
        }
        let publiciseFilter = {
            where: {
                id: queryInfo.publicise_id
            },
        }
        let lotteryPublicise = await ctx.realestateModel.CdLotteryPublicise.findOne(publiciseFilter);
        ret.status = 1;
        ret.data = {
            times: lotteryPublicise.times,
            sort: lotteryPublicise.select_order,
            no: lotteryPublicise.notarization_no,
            house_name: lotteryPublicise.house_name,
        }
        return ret;
    }


}

module.exports = LotteryPubliciseService;
