
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const HOUSE_TYPE = [
    { name: '不限', min: 0, max: 0, value: 0, },
    { name: '1室', min: 1, max: 1, value: 1, },
    { name: '2室', min: 2, max: 2, value: 2, },
    { name: '3室', min: 3, max: 3, value: 3, },
    { name: '4室', min: 4, max: 4, value: 4, },
    { name: '5室', min: 4, max: 4, value: 5, },
    { name: '5室以上', min: 5, max: 10000, value: 6, },
    { name: '别墅', min: 10000, max: 10000000, value: 7, },
]

class NewHouseService extends Service {

    /**
     * 根据不同筛选获取列表
     * @param {*} type 四种默认的按钮筛选 全部all 在售sale 最近开盘open 优惠favourable 首页home
     */
    async getNewHouseListByType(type) {
        const { ctx } = this;
        let types = ['all', 'sale', 'open', 'favourable', 'home'];
        if (!types.includes(type)) {
            ctx.failed('error list type');
        }
        let newHouseList = [];
        if (type === 'home') {
            //为您推荐只展示50条在售楼盘的数据，根据排序序号取数，数字小的排在前面，数字一样的情况下根据时间逆序排列，时间也一样的情况下随机排列；
            let condition = {
                pageSize: 50,
                orderConditions: [{
                    key: 'orderNum',
                    orderSequence: 'asc',
                },
                {
                    key: 'createdAt',
                    orderSequence: 'desc',
                },]
            };
            newHouseList = await this.getNewHouseByFilter(condition);
        } else if (type === 'all') {

        } else if (type === 'sale') {

        } else if (type === 'open') {
            //只展示最近三个月内开盘的楼盘，往前追溯三个月,列表单次加载30条楼盘数据，滑到底部再次加载30条
            let endDate = moment().subtract(30, 'days').format('YYYY-MM-DD HH:mm:ss');
            let condition = {
                queryConditions: [{
                    key: "openDate",
                    value: endDate,
                    operator: "greater"
                }],
            };
            newHouseList = await this.getNewHouseByFilter(condition);
        } else if (type === 'favourable') {
            //点击优惠好盘只展示有优惠的楼盘,列表单次加载30条楼盘数据，滑到底部再次加载30条
            let condition = {
                queryConditions: [{
                    key: "favourableInfo",
                    value: false,
                    operator: "isnull"
                }],
            };
            newHouseList = await this.getNewHouseByFilter(condition);
        }
        return newHouseList;
    }

    /**
     * 新房信息
     * @param {string} id 新房id
     */
    async getNewHouse(id) {
        const { ctx, service } = this;
        //房产基本信息
        let newHouseInfo = await service.houseCommon.newHouse.one(id);
        let newHouseFormat = await this.formatNewHouse(newHouseInfo);
        //图片
        let filter = {
            pageIndex: 1,
            pageSize: 999,
            queryConditions: [{
                key: "state",
                value: '1',
                operator: "equal"
            }, {
                key: "type",
                value: '1',
                operator: "equal"
            }, {
                key: "connectId",
                value: id,
                operator: "equal"
            }],
            orderConditions: [],
        }
        let newHouseImages = await service.houseCommon.houseImage.all(filter);
        ctx.logger.info(newHouseImages);

        let images = [];
        if (newHouseImages.rowCount > 0) {
            for (let i in newHouseImages.results) {
                let image = {
                    id: newHouseImages.results[i].id,
                    path: newHouseImages.results[i].path,
                    description: newHouseImages.results[i].description,
                }
                images.push(image);
            }
        }
        //户型信息
        let newHouseType = await this.getNewHouseType(id);
        let ret = {
            house_basic: newHouseFormat,
            house_images: images,
            house_types: newHouseType
        }
        //TODO对数据格式化处理
        return ret;
    }

    async formatNewHouse(data) {
        const { ctx, service } = this;
        let ret = {};
        if (Object.keys(data).length > 0) {
            let residentialInfo = await service.houseCommon.residential.one(data.residentialId);
            // let developerInfo = await service.houseCommon.developer.one(data.residentialId);
            ret.basic = {
                id: data.id,
                name: data.name,//标题
                description: data.description,//描述
                tags: data.tags.split(','),//项目特色
                favourable_info: data.favourableInfo,//优惠
                reference_avg_price: data.referenceAvgPrice,//均价
                reference_total_price: data.referenceTotalPrice,//总价
                //物业类型
            }

            ret.info = {
                open_date: data.openDate,//开盘时间
                due_date: data.dueDate,//交房时间
                house_area: data.houseArea,//建筑面积
                //装修情况
                green_ratio: data.greenRatio,//绿化
                area_ratio: data.areaRatio,//容积
                address: data.address,//地址
            }
            ret.detail = {
                basic: {
                    reference_avg_price: data.referenceAvgPrice,//均价
                    tags: data.tags.split(','),//项目特色
                    //物业类型
                    property_right_years: residentialInfo.propertyRightYears || '',//产权年限
                    //建筑类型
                    //装修情况
                },
                sale: {
                    //销售状态
                    address: data.address,//地址
                    //售楼地址
                    //开发商
                    open_date: data.openDate,//开盘时间
                    due_date: data.dueDate,//交房时间
                },
                residential: {
                    //占地面积
                    //建筑面积
                    area_ratio: data.areaRatio,//容积率
                    green_ratio: data.greenRatio,//绿化率
                    property_fee: residentialInfo.PropertyFee || '',//物业费用
                    property_company: residentialInfo.propertyCompany || '',//物业公司
                    plan_rooms: data.planRooms,//规划户数
                    plan_parks: data.planParks,//车位情况
                    heat_supply_type: residentialInfo.heatSupplyType || '',//供暖方式
                    water_supply_type: residentialInfo.waterSupplyType || '',//供水方式
                    power_supply_type: residentialInfo.powerSupplyType || '',//供电方式
                    near_by: residentialInfo.nearBy || '',//周边规划
                },
                permit: {
                    pre_sale_permit: data.preSalePermit,//预售证
                    pre_sale_permit_date: data.preSalePermitDate,//发证时间
                    building_numbers: data.buildingNumbers,//绑定楼栋
                }
            }
        }
        return ret;
    }

    /**
     * 通过NewHouseId获取新房户型信息
     * @param {*} id 
     */
    async getNewHouseType(id) {
        const { ctx, service } = this;
        let filter = {
            pageIndex: 1,
            pageSize: 999,
            queryConditions: [{
                key: "state",
                value: 1,
                operator: "equal"
            }, {
                key: "newHouseId",
                value: id,
                operator: "equal"
            }],
            orderConditions: [],
        }
        let newHouseTypeList = await service.houseCommon.newHouseType.all(filter);
        //数据格式化
        let retList = [];
        if (newHouseTypeList.rowCount > 0) {
            let taskList = [];
            for (let i in newHouseTypeList.results) {
                taskList[i] = this.formatNewHouseType(newHouseTypeList.results[i]);
            }

            retList = await Promise.all(taskList).then(result => {
                return result;
            }).catch(error => {
                ctx.failed(error);
            });
        }

        let typeFormat = {};
        for (let i in retList) {
            let type = retList[i].type;
            if (!typeFormat.hasOwnProperty(type)) {
                typeFormat[type] = {
                    name: HOUSE_TYPE[type].name,
                    results: [],
                    count: 0
                };
            }
            typeFormat[type].results.push(retList[i]);
            typeFormat[type].count++;
        }
        let ret = {
            all: {
                results: retList,
                count: retList.length
            }
        };
        Object.assign(ret, typeFormat);
        return ret;
    }

    /**
     * newHouseType数据整理
     * @param {object} data 
     */
    async formatNewHouseType(data) {
        const { ctx } = this;
        let ret = {};
        if (Object.keys(data).length > 0) {
            ret = {
                id: data.id,
                newHouseId: data.newHouseId,
                type: data.type,//类型，如三居、四居
                image: data.image,
                house_type: data.houseType,//住宅类型，如住宅，公寓
                apartment: data.apartment,//户型，如4室2厅3卫
                sale_type: data.saleType,//销售类型，0未开售，1预售，2在售，3售罄
                area: data.area,//面积
                orientation: data.orientation,//朝向
                num: data.num,//房源数量
                price: data.price,//价格
                discount: data.discount,//折扣
                // tag: data.tags.split(','),//特点
            }
        }
        return ret;
    }

    /**
     * 根据筛选条件来获取newHouse
     * @param {*} condition 
     */
    async getNewHouseList(condition) {
        const { ctx, service } = this;
        let page = Number(condition.page) || 1;
        let pageSize = Number(condition.page_size) || 30;
        let filter = {
            pageIndex: page,
            pageSize: pageSize,
            queryConditions: [{
                key: "state",
                value: 1,
                operator: "equal"
            }],
            orderConditions: [
                // {//根据楼盘在售的在前、预售的在后、售罄的在最后
                //     key: 'saleType',
                //     orderSequence: 'desc',
                // }
            ],
        }
        //是否有值来增加筛选项
        if (condition.unit_price) {//单价
            filter.queryConditions.push(
                {
                    key: 'referenceAvgPrice',
                    value: condition.unit_price.min,
                    operator: 'greaterEqual',
                },
                {
                    key: 'referenceAvgPrice',
                    value: condition.unit_price.max,
                    operator: 'lessEqual',
                }
            );
        }
        if (condition.total_price) {//总价
            filter.queryConditions.push(
                {
                    key: 'referenceTotalPrice',
                    value: condition.total_price.min,
                    operator: 'greaterEqual',
                },
                {
                    key: 'referenceTotalPrice',
                    value: condition.total_price.max,
                    operator: 'lessEqual',
                }
            );
        }
        if (condition.area) {//面积
            filter.queryConditions.push(
                {
                    key: 'area',
                    value: condition.area.min,
                    operator: 'greaterEqual',
                },
                {
                    key: 'area',
                    value: condition.area.min,
                    operator: 'lessEqual',
                })
                ;
        }
        if (condition.type) {//户型
            filter.queryConditions.push({
                key: 'Apartment',
                value: condition.type,
                operator: 'equal',
            });
        }
        let newHouseList = await service.houseCommon.newHouse.all(filter);
        let list = [];
        if (newHouseList.rowCount > 0) {
            for (let i in newHouseList.results) {
                let tmp = {
                    id: newHouseList.results[i].id,
                    name: newHouseList.results[i].name,
                    address: newHouseList.results[i].address,
                    tags: newHouseList.results[i].tags.split(','),
                    image: newHouseList.results[i].image,
                    price: newHouseList.results[i].referenceAvgPrice,
                };
                list.push(tmp);
            }
        }
        let ret = {
            results: list,
            count: list.length
        };
        return ret;
    }

    async getNewHouseByFilter(condition) {
        const { ctx, service } = this;
        let page = Number(condition.page) || 1;
        let pageSize = Number(condition.pageSize) || 30;
        let filter = {
            pageIndex: page,
            pageSize: pageSize,
            queryConditions: [{
                key: "state",
                value: 1,
                operator: "equal"
            },],
            orderConditions: [],
        }
        if (condition.hasOwnProperty('queryConditions')) {
            filter.queryConditions = filter.queryConditions.concat(condition.queryConditions);
        }
        if (condition.hasOwnProperty('orderConditions')) {
            filter.orderConditions = filter.orderConditions.concat(condition.orderConditions);
        }
        let newHouseList = await service.houseCommon.newHouse.all(filter);
        let list = [];
        if (newHouseList.rowCount > 0) {
            for (let i in newHouseList.results) {
                let tmp = {
                    id: newHouseList.results[i].id,
                    name: newHouseList.results[i].name,
                    address: newHouseList.results[i].address,
                    tags: newHouseList.results[i].tags.split(','),
                    image: newHouseList.results[i].image,
                    price: newHouseList.results[i].referenceAvgPrice,
                };
                list.push(tmp);
            }
        }
        let ret = {
            results: list,
            count: list.length
        };
        return ret;
    }

}

module.exports = NewHouseService;
