
'use strict';

const Service = require('egg').Service;
const R = require('ramda');

class RentalHouseService extends Service {

    async getRentalHousesByFilter(condition) {

        const { ctx } = this;

        const queryConditions = [];
        if (condition.brand) {
            queryConditions.push({
                key: 'developerId',
                value: condition.brand,
                operator: 'equal',
            });
        }
        if (condition.price && condition.price.max) {
            queryConditions.push({
                key: 'price',
                value: condition.price.min,
                operator: 'greaterEqual',
            });
            queryConditions.push({
                key: 'price',
                value: condition.price.max,
                operator: 'lessEqual',
            });
        }
        if (condition.house_type) {
            queryConditions.push({
                key: 'rentalHouseType',
                value: condition.house_type,
                operator: 'equal',
            });
        }
        queryConditions.push({
            key: "state",
            value: 1,
            operator: "equal"
        });
        const filter = {
            pageSize: 30,
            queryConditions: queryConditions,
            orderConditions: [{
                key: 'price',
                orderSequence: 'asc',
            },],
        }
        const rental_houses_results = await ctx.service.houseCommon.rentalHouse.all(filter);
        let rental_houses = await this.formatRentalHouseBasic(rental_houses_results.results);
        rental_houses = R.project(['id', 'image', 'name', 'address', 'price', 'tags', 'favourable_info'])(rental_houses);
        return rental_houses;
    }

    //房源详细信息
    async getRentalHouse(rental_house_id) {

        const { ctx } = this;

        //获取房源基本信息
        let house_basic_result = await ctx.service.houseCommon.rentalHouse.one(rental_house_id);
        if (!house_basic_result || !house_basic_result.id) {
            ctx.failed('house not found');
        }

        rental_house_id = house_basic_result.id;

        //获取房源图片信息
        const p_house_images = this.getRentalHouseImages(rental_house_id);

        //获取房源类型
        const p_house_types = this.getRentalHouseTypes(rental_house_id);

        const p_ret = await Promise.all([p_house_images, p_house_types]).then(result => {//等待所有异步内容获取完成
            return result;
        }).catch(error => {
            ctx.failed(error);
        });

        const house_images = p_ret[0];
        const house_types = p_ret[1];

        //处理房源基本信息格式
        let house_basic = await this.formatRentalHouseBasic([house_basic_result]);
        house_basic = house_basic[0];

        ctx.service.house.footPrint.addFootPrint({ id: rental_house_id, type: 2 });//添加用户足迹记录

        return { house_basic, house_images, house_types };
    }

    //获取房源画册
    async getRentalHouseImages(rental_house_id) {

        const { ctx } = this;

        const house_image_filter = {
            pageSize: 10000,
            queryConditions: [
                {
                    key: 'state',
                    value: 1,
                    operator: 'equal'
                },
                {
                    key: 'type',
                    value: 0,
                    operator: 'equal'
                },
                {
                    key: 'connectId',
                    value: rental_house_id,
                    operator: 'equal'
                },
            ],

        };
        const house_images_results = await ctx.service.houseCommon.houseImage.all(house_image_filter);
        const house_images = R.project(['id', 'path', 'description'])(house_images_results.results);

        return house_images;
    }

    //获取房源类型
    async getRentalHouseTypes(rental_house_id) {

        const { ctx } = this;

        const house_type_filter = {
            pageSize: 1000,
            queryConditions: [
                {
                    key: 'rentalHouseId',
                    value: rental_house_id,
                    operator: 'equal',
                },
                {
                    key: 'state',
                    value: '1',
                    operator: 'equal',
                },
            ]
        }
        const house_types_results = await ctx.service.houseCommon.rentalHouseType.all(house_type_filter);
        const house_types = house_types_results.results;
        let house_type_image_filter = {
            pageSize: 10000,
            queryConditions: [
                {
                    key: 'state',
                    value: 1,
                    operator: 'equal'
                },
                {
                    key: 'type',
                    value: 3,
                    operator: 'equal',
                },
            ],

        };
        const ret = {};
        for (let i in house_types) {
            const house_type = house_types[i];
            house_type_image_filter.queryConditions.push({
                key: 'connectId',
                value: house_type.id,
                operator: 'equal',
            });
            const house_type_images_results = await ctx.service.houseCommon.houseImage.all(house_type_image_filter);
            if (!ret[house_type.name] || ret[house_type.name].length === 0) {
                ret[house_type.name] = [];
            }
            ret[house_type.name].push({
                id: house_type.id,
                name: house_type.name,
                area: house_type.area,
                price: house_type.price,
                images: R.project(['id', 'path', 'description'])(house_type_images_results.results)
            });
        }

        return ret;
    }


    async formatRentalHouseBasic(rental_houses) {

        const { ctx } = this;
        if (!rental_houses || !Array.isArray(rental_houses) || rental_houses.length === 0) {
            return [];
        }
        const ret = [];
        for (let i in rental_houses) {
            const rental_house = rental_houses[i];
            const temp_rental_house = {
                id: rental_house.id,
                name: rental_house.name,//房源名称
                image: rental_house.imgae,//列表展示图片
                residential_id: rental_house.residentialId,//小区id
                address: rental_house.address,//详细地址
                price: rental_house.price,//价格
                discount: rental_house.discount,//折扣
                // business_license: rental_house.businessLicense,
                tags: rental_house.tags.split(','),//房源特点
                favourable_info: rental_house.favourableInfo,//优惠信息
                introduce: rental_house.description || `该社区亮点：
                1、房源介绍：精装公寓，家电齐全，独立卫生间，南通北透，独门独户。
                2、房型说明：独门独户，拎包入住，满足各种需求，采光
                3、社区配套：健身房、休憩区、小影院、娱乐室、商务区、无人商店、快递柜。            
                4、房间配套：空调、Wifi、洗衣机、热水器、冰箱、床、衣柜、电脑桌、沙发。`,
                room_configuration: JSON.parse(rental_house.roomConfiguration),//房间配置
                notice: rental_house.notice,//入住须知
            };
            ret.push(temp_rental_house);
        }

        return ret;
    }

}

module.exports = RentalHouseService;
