
'use strict';

const Service = require('egg').Service;
const R = require('ramda');
const _ = require('lodash');
const moment = require('moment');
class InstitutionService extends Service {


    //机构列表页
    async getInstitutions(input) {

        const { ctx } = this;
        const user_uuid = ctx.userUuid;
        const { cats, ages, lat, lng, address } = input;
        const where_institutions = input.institutions;
        //保存定位记录
        if (address && lat && lng) {
            ctx.classModel.CourseLogUserGps.add({ user_uuid, address, lat, lng });
        }

        //处理条件过滤条件
        let where = { status: 1, is_deleted: 0 };
        if (cats) {
            const cat_ret = await ctx.classModel.CourseCat.one({ where: { id: cats } });
            const cat_id = cat_ret.id;
            const cat_level = cat_ret.level;
            const next_level = cat_level + 1;
            const next_next_level = cat_level + 2;
            const next_cat = await ctx.classModel.CourseCat.all({ where: { level: next_level, parent_id: cat_id } });
            const next_next_cat = await ctx.classModel.CourseCat.all({ where: { level: next_next_level, first_id: cat_id } });
            let cat_ids = [cat_id,];
            cat_ids = cat_ids.concat(R.pluck('id', next_cat)).concat(R.pluck('id', next_next_cat));
            const institutions = await ctx.classModel.CourseInstitutionToCat.all({ where: { cat_id: { $in: cat_ids } } });
            where.id = { $in: R.pluck('institution_id', institutions) };
        }
        if (ages) {
            where.max_age = { $gte: ages };
            where.min_age = { $lte: ages };
        }
        if (where_institutions) {
            where.corner = { $ne: '' };
        }

        //关联校区表查找信息
        const include = [{ model: ctx.classModel.CourseArea, where: { status: 1, is_deleted: 0 }, attributes: ['id', 'institution_id', 'name', 'address', 'lat', 'lng'] }];
        const attributes = ['id', 'name', 'type', 'establishment_time', 'class_type', 'teacher_count', 'teacher_experience', 'corner', 'min_age', 'max_age', 'price', 'characteristic',];
        const institutions = await ctx.classModel.CourseInstitution.findAll({ attributes, include, where });

        const institution_area_list = await this.getInstitutionAreaList(institutions);//将校区表处理成数组
        // const from = { lat, lng };
        const area_lbs = await this.computeDistance(institution_area_list, { lat, lng });//计算距离信息
        const institution_areas = await this.findShortestDistanceAreas(institution_area_list, area_lbs);//查找最短距离并输出
        const ret = await this.formatInstitutions(institution_areas);

        return ret;
    }

    /**
     * 机构详情页
     */
    async getInstitution(input) {

        const { ctx } = this;
        const user_uuid = ctx.userUuid;
        const { institution_id, area_id } = input;
        const lat = ctx.isEmpty(input.lat) ? 0 : input.lat;
        const lng = ctx.isEmpty(input.lng) ? 0 : input.lng;
        const institution = await ctx.classModel.CourseInstitution.one({ where: { id: institution_id } });
        const teachers = await this.getTeachers({ institution_id, limit: 6 });
        const classes = await this.getClasses({ institution_id, limit: 4 });

        const institution_images = await ctx.classModel.CourseImages.all({ where: { type: 1, type_id: institution_id } });//图片

        //处理详情页信息
        const current_area = await ctx.classModel.CourseArea.one({ id: area_id });
        let institution_detail = await this.formatInstitutions([institution]);
        institution_detail = institution_detail[0];
        institution_detail.address = current_area.address;
        institution_detail.phone = current_area.phone;
        institution_detail.description = institution.description;
        institution_detail.point = institution.point;
        institution_detail.honor = institution.honor;
        institution_detail.is_collect = 0;

        //计算校区距离
        const areas_ret = await this.getInstitutionAreas({ institution_id, limit: 1000 });//校区
        const area_rows = areas_ret.rows;
        const area_lbs = (lat > 0 || lng > 0) ? await this.computeDistance(area_rows, { lat, lng }) : [];//计算距离信息
        const areas = [];

        for (let i in area_rows) {

            const area = area_rows[i];
            const lbs = i < area_lbs.length ? area_lbs[i] : { distance: 0, duration: 0, travel_method: '' };
            const area_name = area.name;
            const distance = lbs.distance;
            const duration = lbs.duration;

            let travel_tips = `距我${distance}km,开车${duration}分钟`;
            if (lbs.travel_method === 'walking') {
                travel_tips = `距我${distance}km,步行${duration}分钟`;
            }
            if (distance === 0 && duration === 0) {
                travel_tips = '暂无法计算出距离';
            }
            if (area.id == area_id && area.institution_id == institution_id) {
                institution_detail.travel_tips = travel_tips;
                institution_detail.lat = area.lat;
                institution_detail.lng = area.lng;
            }
            areas.push({
                id: area.id,
                institution_id: area.institution_id,
                name: area_name,
                address: area.address,
                phone: area.phone,
                travel_tips,
                lat: area.lat,
                lng: area.lng,
            });
        }

        //是否收藏过
        const collection_ret = await ctx.classModel.CourseUserCollection.one({ where: { user_uuid, institution_id, is_deleted: 0 } });
        if (collection_ret && collection_ret.id) {
            institution_detail.is_collect = 1;
        }


        //处理图片
        const photo_album = [];
        for (let i in institution_images) {
            const institution_image = institution_images[i];
            photo_album.push({
                image: institution_image.image_url,
                is_video: institution_image.is_video,
                video_url: institution_image.video_url,
            });
        }
        institution_detail.photo_album = photo_album;

        return { institution_detail, teachers: teachers.rows, classes: classes.rows, areas: areas };
    }

    async getTeacher(teacher_id) {

        const { ctx } = this;
        const where = { id: teacher_id };
        let teacher = await ctx.classModel.CourseTeacher.one({ where });
        teacher.dataValues.point_tags = teacher.point ? teacher.point.split(';') : [];
        teacher.dataValues.work_experience_tags = teacher.work_experience ? teacher.work_experience.split(';') : [];

        const institution = await ctx.classModel.CourseInstitution.find({ where: { id: teacher.institution_id } });
        teacher.dataValues.institution_name = institution.name;

        return teacher;
    }

    async getClass(class_id) {

        const { ctx } = this;
        const where = { id: class_id };
        let classs = await ctx.classModel.CourseClass.one({ where });
        classs.dataValues.age_text = `${classs.min_age}-${classs.max_age}岁`;
        classs.dataValues.point_tags = classs.point ? classs.point.split(';') : [];
        classs.dataValues.photo_album = classs.image ? classs.image.split(';') : [];

        return classs;
    }

    async getTeachers(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'avatar', 'teacher_experience', 'lesson', 'educational_background', 'certificate'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const teachers = await ctx.classModel.CourseTeacher.list({ attributes, page, limit, where });

        const ret = [];
        for (let i in teachers) {
            const teacher = teachers[i];
            ret.push({
                id: teacher.id,
                institution_id: teacher.institution_id,
                name: teacher.name ? teacher.name : '',
                avatar: teacher.avatar ? teacher.avatar : '',
                teacher_experience: teacher.teacher_experience ? teacher.teacher_experience : '',
                lesson: teacher.lesson ? teacher.lesson : '',
                educational_background: teacher.educational_background ? teacher.educational_background : '',
                certificate: teacher.certificate ? teacher.certificate : '',
            });
        }

        return ret;

    }

    async getClasses(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'type', 'price'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const classes = await ctx.classModel.CourseClass.list({ attributes, page, limit, where });
        const p_class_images = [];

        //获取图片信息
        for (let j in classes.rows) {
            const classs = classes.rows[j];
            p_class_images[j] = ctx.classModel.CourseImages.one({ where: { type: 2, type_id: classs.id, is_cover: 1, is_video: 0 } });
        }
        const class_images = await Promise.all(p_class_images).then(result => {//等待所有异步内容获取完成
            return result;
        }).catch(error => {
            ctx.failed(error);
        });

        //格式化课程信息
        let ret = [];
        for (let i in classes.rows) {
            let classs = classes.rows[i];
            classs.dataValues.image = class_images[i] ? class_images[i].image_url : '';;
            classs.dataValues.price_text = classs.price ? classs.price : '现场咨询';
            ret.push(classs);
        }

        return { page: classes.page, count: classes.count, rows: ret };

    }

    async getInstitutionAreas(input) {

        const { ctx } = this;
        const attributes = ['id', 'institution_id', 'name', 'address', 'phone', 'lat', 'lng'];
        const { institution_id, page, limit } = input;
        const where = { institution_id };
        const areas = await ctx.classModel.CourseArea.list({ attributes, page, limit, where });

        return areas;

    }

    //机构列表 并有多校区 {id：1，..., course_areas:[{},{}]
    async getInstitutionAreaList(institutions) {

        const { ctx } = this;

        if (!Array.isArray(institutions) || institutions.length === 0) {
            return [];
        }

        const ret = [];
        for (let i in institutions) {
            let institution = institutions[i];
            if (!institution.course_areas || institution.course_areas.length === 0) continue;
            const course_areas = institution.course_areas;
            for (let j in course_areas) {
                const course_area = course_areas[j];
                ret.push({
                    id: institution.id,
                    name: institution.name,
                    type: institution.type,
                    image: institution.image,
                    establishment_time: institution.establishment_time,
                    class_type: institution.class_type,
                    teacher_count: institution.teacher_count,
                    teacher_experience: institution.teacher_experience,
                    corner: institution.corner,
                    min_age: institution.min_age,
                    max_age: institution.max_age,
                    price: institution.price,
                    characteristic: institution.characteristic,
                    area_id: course_area.id,
                    area_name: course_area.name,
                    area_address: course_area.address,
                    lat: course_area.lat,
                    lng: course_area.lng,
                });
            }
        }

        return ret;
    }

    /**
     * 起点（）到多个终点（[{ lng: 120.212997, lat: 30.29133 }];）  计算距离 distance=3km
     * @param {*} lbs_array [{ lng: 120.212997, lat: 30.29133 }]
     * @param {*} from_gps { lng: 120.069206, lat: 30.291121 }
     * @param {*} distance 3
     */
    async computeDistance(lbs_array, from_gps, distance = 3) {

        const { ctx } = this;

        if (!Array.isArray(lbs_array) || lbs_array.length === 0) {
            return [];
        }

        const from = from_gps;
        const lat = from.lat;
        const lng = from.lng;
        // console.info({ lat, lng });
        //计算距离
        const driving_results = await ctx.service.course.lbs.getLBSDistance('driving', { lat, lng }, lbs_array);
        const walking_results = await ctx.service.course.lbs.getLBSDistance('walking', { lat, lng }, lbs_array);
        if (!walking_results.results && !driving_results.results) {
            return [];
        }

        //处理距离结果
        const drivings = driving_results.results.map(item => {
            const distance = (item.distance / 1000).toFixed(1);
            const duration = Math.ceil(item.duration / 60);
            return { distance: distance, duration: duration }
        });
        const walkings = walking_results.results.map(item => {
            const distance = (item.distance / 1000).toFixed(2);
            const duration = Math.ceil(item.distance / 80);
            return { distance: distance, duration: duration }
        });

        //格式化 判断步行或驾车
        const results = [];
        for (let i in lbs_array) {
            let lbs = lbs_array[i];
            const driving = (drivings[i] && drivings[i].distance !== -1) ? drivings[i] : { distance: 0, duration: 0 };
            const walking = (walkings[i] && walkings[i].distance !== -1) ? walkings[i] : { distance: 0, duration: 0 };
            const lbs_distance = driving.distance > distance ? driving : walking;
            const travel_method = driving.distance > distance ? 'driving' : 'walking';
            results.push({
                lat: lbs.lat,
                lng: lbs.lng,
                distance: lbs_distance.distance,
                duration: lbs_distance.duration,
                travel_method,
            });
        }

        return results;

    }

    /**
     * 
     * @param {*} institution_areas 校区列表
     * @param {*} areas_list 校区距离列表
     */
    async findShortestDistanceAreas(institution_areas, areas_list) {

        const { ctx } = this;

        let ret = [];
        // const institution_lbs = await this.computeDistance(areas_list);

        const institution_lbs = areas_list;
        for (let i in institution_areas) {

            let institution_area = institution_areas[i];
            //TODO 校验经纬度
            const lbs = i < institution_lbs.length ? institution_lbs[i] : { distance: 0, duration: 0, travel_method: '' };

            const area_name = institution_area.area_name;
            const distance = lbs.distance;
            const duration = lbs.duration;
            let travel_tips = `${area_name},距我${distance}km,开车${duration}分钟`;
            if (lbs.travel_method === 'walking') {
                travel_tips = `${area_name},距我${distance}km,步行${duration}分钟`;
            }
            if (distance === 0 && duration === 0) {
                travel_tips = '暂无法计算出距离';
            }
            // area.distance = distance;
            // area.duration = duration;
            // area.travel_tips = travel_tips;

            institution_area.distance = distance;
            institution_area.duration = duration;
            institution_area.travel_tips = travel_tips;

            ret[i] = institution_area;
        }

        const results = [];
        ret = _.orderBy(ret, ['distance'], ['asc']);
        for (let j in ret) {
            const val = ret[j];
            if (!results[val.id]) {
                results[val.id] = val;
            }
        }

        return results;
    }

    async formatInstitutions(institutions) {

        const { ctx } = this;
        if (!Array.isArray(institutions) || institutions.length === 0) {
            return [];
        }

        //处理机构图片
        const p_institution_images = [];
        for (let j in institutions) {
            const institution = institutions[j];
            p_institution_images[j] = ctx.classModel.CourseImages.one({ where: { type: 1, type_id: institution.id, is_cover: 1, is_video: 0 } });
        }
        const institution_images = await Promise.all(p_institution_images).then(result => {//等待所有异步内容获取完成
            return result;
        }).catch(error => {
            ctx.failed(error);
        });

        const ret = [];
        for (let i in institutions) {
            const institution = institutions[i];
            // const course_areas = institution.course_areas;
            // if (!course_areas || course_areas.length === 0) continue;
            const age_tag = institution.min_age + '-' + institution.max_age + '岁';
            const build_time = moment().format('YYYY') - institution.establishment_time;
            const tags = [];
            if (institution.max_age) {
                tags.push(age_tag);
            }
            if (institution.class_type) {
                tags.push(institution.class_type);
            }
            if (build_time) {
                tags.push('成立' + build_time + '年');
            }
            const image = institution_images[i] ? institution_images[i].image_url : '';

            ret.push({
                id: institution.id,
                name: institution.name,
                image,
                establishment_time: institution.establishment_time,
                class_type: institution.class_type,
                teacher_count: institution.teacher_count ? institution.teacher_count : 0,
                teacher_experience: institution.teacher_experience ? institution.teacher_experience : 0,
                corner: institution.corner ? institution.corner : '',
                min_age: institution.min_age,
                max_age: institution.max_age,
                price: institution.price ? institution.price : 0,
                tags,
                area_id: institution.area_id,//校区id TODO
                travel_tips: institution.travel_tips, //TODO
                characteristic: institution.characteristic ? institution.characteristic : '',
            });
        }

        return ret;
    }







}

module.exports = InstitutionService;
