
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const _ = require('lodash');
const TypeConfig = {
    blacklist: 1,
    callrisk: 2
}
const TypeConfigFlip = {
    1: 'blacklist',
    2: 'callrisk',
}
class ApplyService extends Service {


    /**
     * 进入黑名单查询页面
     */
    async blacklistInit() {
        const { ctx } = this;
        let ret = {
            have_be_pay_order: false,
            order_id: null,
            phone: '',
            placeholder: {
                name: '',
                id_card: ''
            }
        }
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let bePayOrder = await ctx.service.credit.order.getBePayOrder('blacklist');
        if (bePayOrder.length !== 0) {
            ret.have_be_pay_order = true;
            ret.order_id = bePayOrder[0].id;
            ret.phone = bePayOrder[0].phone.substring(0, 3) + '****' + bePayOrder[0].phone.substring(7, 11)
        }

        //第一次查询成功的订单,反显姓名和身份证
        let filter = {
            where: {
                user_id: ctx.userId,
                pay_status: 1,
                state: '已支付',
                type: 1,
                valid: 1,
            },
            order: [['id', 'asc']]
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(filter);
        if (orderInfo != null) {
            ret.placeholder.name = orderInfo.name;
            ret.placeholder.id_card = orderInfo.id_card;
        }
        return ret;
    }

    /**
     * 进入个人通话查询页面
     */
    async callriskInit() {

        const { ctx } = this;
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let ret = {
            have_be_pay_order: false,
            order_id: null,
            phone: '',
            placeholder: {
                name: '',
                id_card: '',
            }
        }
        let bePayOrder = await ctx.service.credit.order.getBePayOrder('callrisk');
        if (bePayOrder.length !== 0) {
            ret.have_be_pay_order = true;
            ret.order_id = bePayOrder[0].id;
            ret.phone = bePayOrder[0].phone.substring(0, 3) + '****' + bePayOrder[0].phone.substring(7, 11)
        }

        //第一次查询成功的订单,反显姓名和身份证
        let filter = {
            where: {
                user_id: ctx.userId,
                pay_status: 1,
                state: '已支付',
                type: 2,
                valid: 1,
            },
            order: [['id', 'asc']]
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(filter);
        if (orderInfo && orderInfo.id) {
            ret.placeholder.name = orderInfo.name;
            ret.placeholder.id_card = orderInfo.id_card;
        }
        return ret;
    }


    /**
     * 获取短信验证码
     * @param {*} inputParams 
     */
    async getVerificationCode(inputParams) {
        const { ctx, app } = this;
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let ret = {
            code: 1,
            msg: '短信发送成功',
        }
        ctx.logger.info(inputParams);
        //姓名手机号和身份证校验
        let idVerify = ctx.helper.verify_id_card(inputParams.id_card);
        let phoneVerify = ctx.helper.isPhoneNumber(inputParams.phone);
        let nameVerify = ctx.helper.verify_real_name(inputParams.name);
        if (!idVerify) {
            ctx.failed('身份证输入有误，请校验后输入');
        }
        if (!phoneVerify) {
            ctx.failed('手机号输入有误，请校验后输入');
        }
        if (nameVerify !== 1) {
            ctx.failed('姓名输入有误，请校验后输入');
        }

        //当用户在首笔“黑名单检测报告”或“个人通话检测报告”支付成功后，当用户查询其他不同的2要素(姓名,身份证)信息时:仅供本人查询使用，无法查询其他人信息
        let orderFilter = {
            arrtibutes: ['id', 'name', 'id_card'],
            where: {
                user_id: ctx.userId,
                state: '已支付',
                type: 1,
                valid: 1
            }
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(orderFilter);
        if (orderInfo !== null && (inputParams.name !== orderInfo.name || inputParams.id_card !== orderInfo.id_card)) {
            ctx.failed('仅供本人查询使用，无法查询其他人信息');
        }

        //数盒魔方三要素校验
        await ctx.service.credit.common.shuhemofangCheck('blacklist', inputParams);

        //验证码
        //60秒只能获取一次
        let lockKey = 'mine:credit:black:lock' + inputParams.phone;
        let lock = await app.redis.get(lockKey);
        ctx.logger.info(lock);
        if (lock && lock !== null) {
            ctx.failed('获取验证码过于频繁，请稍后再试');
        }
        await app.redis.set(lockKey, 'lock', 'EX', 60);//60秒锁

        //每日次数限制
        let timesKey = 'mine:credit:black:times' + inputParams.phone;
        let times = await app.redis.get(timesKey);
        ctx.logger.info(times);
        if (!times) {
            times = 0;
        }
        times++;
        if (times >= 11) {
            ctx.failed('今日获取验证码次数过多，请明日再试');
        }
        let pexpire = new Date(moment().add(1, 'days').format('YYYY-MM-DD 00:00:00')).getTime();
        let validTime = Number(pexpire) - Number(moment().format('x'));
        ctx.logger.info(validTime);
        await app.redis.set(timesKey, times, 'PX', validTime);//今日有效

        //生成
        let code = '';
        code = Math.round(Math.random() * 9999).toString();
        let len = code.length;
        if ((4 - len) > 0) {
            ß
            for (var i = 0; i < 4 - len; i++) {
                code = '0' + code;
            }
        }
        let codeKey = 'mine:credit:black:code' + inputParams.phone;
        let codeData = {//用于避免用户获取验证码之后修改姓名/身份证等
            code: code,
            name: inputParams.name,
            id_card: inputParams.id_card
        }
        await app.redis.set(codeKey, JSON.stringify(codeData), 'EX', 60 * 5);//5分钟有效时间


        let smsParams = {
            phone: inputParams.phone,
            content: '【51公积金】您本次操作的短信验证码为' + code + '，5分钟内有效。谨慎保管，切勿告诉他人。'
        }
        ctx.logger.info(smsParams);
        let smsResult = await ctx.service.common.sms.sendSms(smsParams);
        if (smsResult.status != 201) {
            ctx.failed('短信发送失败,稍后再试');
        }
        return ret;
    }



    /**
     * 从数据接口获取用户的黑名单报告信息
     * @param {*} inputParams 
     */
    async applyBlacklist(inputParams) {
        const { ctx, app } = this;
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let ret = {
            order_id: null,
            report_id: null,
            second: false,//再次查询的是待支付中的订单
        }
        //验证码校验
        let codeKey = 'mine:credit:black:code' + inputParams.phone;
        let codeData = await app.redis.get(codeKey);
        ctx.logger.info({ codeData: codeData });
        if (codeData === null || !codeData) {
            ctx.failed('请先获取验证码');
        }
        codeData = JSON.parse(codeData);
        if (codeData.code !== inputParams.code) {
            ctx.failed('验证码错误，请重试');
        }
        if (codeData.name !== inputParams.name || codeData.id_card !== inputParams.id_card) {
            ctx.failed('获取验证码后请不要修改姓名、身份证等信息');
        }

        //当用户在首笔“黑名单检测报告”或“个人通话检测报告”支付成功后，当用户查询其他不同的2要素(姓名,身份证)信息时:仅供本人查询使用，无法查询其他人信息
        let orderFilter = {
            arrtibutes: ['id', 'name', 'id_card'],
            where: {
                user_id: ctx.userId,
                state: '已支付',
                type: 1,
                valid: 1
            }
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(orderFilter);
        if (orderInfo !== null && (inputParams.name !== orderInfo.name || inputParams.id_card !== orderInfo.id_card)) {
            ctx.failed('仅供本人查询使用，无法查询其他人信息');
        }

        //判断用户当前三要素是否是二次查询未支付订单
        let bePayOrder = await ctx.service.credit.order.getBePayOrderByThree('blacklist', inputParams);
        if (bePayOrder !== null) {
            ret.second = true;
            ret.order_id = bePayOrder.id;
            ret.report_id = bePayOrder.report_id;
            return ret;
        }

        //生成订单
        let orderData = {
            type: 1,
            name: inputParams.name,
            phone: inputParams.phone,
            id_card: inputParams.id_card
        }
        let order = await ctx.service.credit.order.createOrder(orderData);
        ret.order_id = order.id;

        //调用数据接口获取黑名单数据
        let params = {
            sign: "",
            signParams: {
                appKey: this.config.BLACKLIST_APPLY_APPKEY,
                timestamp: String(new Date().getTime()),
            },
            phone: inputParams.phone,
            name: inputParams.name,
            idcard: inputParams.id_card
        }
        params.sign = await ctx.service.credit.common.sign(params.signParams, this.config.BLACKLIST_APPLY_APPSECRET);
        ctx.logger.info(params);
        let url = this.config.BLACKLIST_APPLY_URL;
        let result = await ctx.helper.send_request(url, params, { method: 'POST' });
        ctx.logger.info('blacklist:apply:result:' + JSON.stringify(result));
        if (result.status != 200) {
            ctx.failed('数据获取接口异常');
        }
        let applyData = {
            type: 1,
            user_id: ctx.userId,
            app_user_id: ctx.appUserId,
            app_id: ctx.appId,
            app_type_id: ctx.appTypeId,
            name: inputParams.name,
            phone: inputParams.phone,
            id_card: inputParams.id_card,
            timestamp: params.signParams.timestamp,
            appkey: this.config.BLACKLIST_APPLY_APPKEY,
            sign: params.sign,
            r_code: result.data.code,
            r_msg: result.data.msg,
            r_order_id: result.data.data.order_id,
        }
        let apply = await ctx.prometheusModel.CreditApply.create(applyData);
        if (result.data.code !== 0) {
            ctx.failed('数据获取异常');
        }
        let blacklistInfo = result.data.data.blackList;
        let report = await ctx.service.credit.blacklist.createReport(inputParams, blacklistInfo);
        ret.report_id = report.id;

        //更新订单
        //生成order_no
        let orderNo = await ctx.service.credit.common.getOrdertNo(TypeConfigFlip[order.type], order.id);
        let updateOrderData = {
            order_no: orderNo,
            apply_id: apply.id,
            report_id: report.id,
            state: '待支付',
            order_time: moment().format('YYYY-MM-DD HH:mm:ss'),
            state_time: moment().format('YYYY-MM-DD HH:mm:ss'),
            valid: 1
        }
        await ctx.prometheusModel.CreditOrder.update(updateOrderData, { where: { id: order.id } });
        await ctx.service.credit.order.logOrder(order.id);

        return ret;
    }

    /**
     * 从数据接口获取用户的个人检测报告信息
     * @param {*} inputParams 
     */
    async applyCallrisk(input) {

        const { ctx } = this;
        const { name, phone, id_card, password, net } = input;
        const user_id = ctx.userId;
        const app_type_id = ctx.appTypeId;
        const app_user_id = ctx.appUserId;
        const app_id = ctx.appId;
        const type = TypeConfig['callrisk'];//个人通话类型
        let report_id = null;
        //数盒魔方三要素校验
        await ctx.service.credit.callrisk.checkThreeElements(input);
        // await ctx.service.credit.common.shuhemofangCheck('callrisk', input);

        //判断用户当前三要素是否是二次查询未支付订单
        const check_second_ret = await this.checkSecond({ name, phone, id_card });
        if (check_second_ret.order_id) {
            return check_second_ret;
        }

        //生成订单
        const order_data = { type, name, phone, id_card, user_id, app_type_id, app_user_id, app_id };
        const order = await ctx.service.credit.order.createOrder(order_data);
        const order_id = order.id;

        //创建yys-order apply 记录
        const apply = await this.createYysApply({ name, phone, id_card, password, net, order_id });
        const order_sn = apply.r_order_id;

        //第一次调用运营商获取验证码接口
        const input_params = {
            phone, name,
            ID: id_card,
            password: apply.service_code,
        };
        const operator_params = apply.operator.params;
        let get_code_params = { orderSn: order_sn, data: {} };
        for (let k in operator_params) {
            const operator_params_val = operator_params[k];
            // if (operator_params_val.name === 'phone') {
            //     continue;
            // }
            get_code_params.data[operator_params_val.name] = input_params[operator_params_val.name];
        }
        // get_code_params.data.phoneCode = '';
        const yys_get_code = await ctx.service.credit.yys.getCode(get_code_params);
        if (yys_get_code.code !== 0) {
            ctx.failed(yys_get_code.msg);
        }
        const order_no = await ctx.service.credit.common.getOrdertNo('callrisk', order_id);
        await ctx.prometheusModel.CreditOrder.update({ order_no, apply_id: apply.id }, { where: { id: order_id } });

        return { order_id, report_id, first: check_second_ret.first, second: check_second_ret.second, order_sn };
    }




    async createYysApply(input) {

        const { ctx } = this;
        const user_id = ctx.userId;
        const app_type_id = ctx.appTypeId;
        const app_user_id = ctx.appUserId;
        const app_id = ctx.appId;
        const { name, phone, id_card, password, net, order_id } = input;

        //运营商API.创建订单号
        const order_sn_info = await ctx.service.credit.yys.getOrderSn({ phone, net });
        if (order_sn_info.code !== 0) {
            ctx.failed('getOrderSn error msg:' + order_sn_info.msg);
        }

        //创建申请订单
        const apply_data = {
            type: 2, user_id, app_user_id, app_id, app_type_id, name, phone, id_card,
            appkey: this.config.YYS_APP_KEY,
            service_code: password,
            r_code: order_sn_info.code,
            r_msg: order_sn_info.msg,
            r_order_id: order_sn_info.data.orderSn,
            operator: JSON.stringify(order_sn_info.data.operator)
        }
        const apply = await ctx.prometheusModel.CreditApply.create(apply_data);

        return apply;
    }

    //判断用户当前三要素是否是二次查询未支付订单
    async checkSecond(input) {
        const { ctx } = this;
        const { name, phone, id_card } = input;
        const { user_id } = this;
        const type = TypeConfig['callrisk'];//个人通话类型

        let second = false;//再次查询的是待支付中的订单
        let order_id = null;
        let report_id = null;

        //当用户在首笔“黑名单检测报告”或“个人通话检测报告”支付成功后，当用户查询其他不同的2要素(姓名,身份证)信息时:仅供本人查询使用，无法查询其他人信息
        let orderFilter = {
            arrtibutes: ['id', 'name', 'id_card'],
            where: {
                user_id: user_id,
                state: '已支付',
                type: type,
                valid: 1
            }
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(orderFilter);
        if (orderInfo !== null && (name !== orderInfo.name || id_card !== orderInfo.id_card)) {
            ctx.failed('仅供本人查询使用，无法查询其他人信息');
        }

        // const order_filter = { arrtibutes: ['id'], where: { user_id, type, valid: 1 } }
        // const order_list = await ctx.prometheusModel.CreditOrder.findAll(order_filter);
        // if (order_list === undefined || order_list.length === 0) {
        //     first = true;
        // }

        //判断用户当前三要素是否是二次查询未支付订单
        const be_pay_order = await ctx.service.credit.order.getBePayOrderByThree('callrisk', { name, phone, id_card });
        if (be_pay_order !== null) {
            second = true;
            order_id = be_pay_order.id;
            report_id = be_pay_order.report_id;
        }

        return { order_id, report_id, second };
    }









}

module.exports = ApplyService;
