'use strict';
const cheerio = require('cheerio');
const Controller = require('egg').Controller;
class ToolController extends Controller {

    //获取地图上的点位
    async getMapPoint() {
        const { ctx } = this;
        const input_parmas = ctx.params;
        const rule = {
            house_style: { type: 'string', required: true },
            area_code: { type: 'string', required: true },
            level: { type: 'string', required: true },
        }
        ctx.validate(rule, input_parmas);

        const house_style = input_parmas.house_style;
        const area_code = input_parmas.area_code;
        const level = input_parmas.level;

        let ret = [];
        if (house_style === 'used_house') {
            ret = await ctx.service.house.tool.getUsedHouseMapPoint(area_code, level);
        } else if (house_style === 'new_house') {
            ret = await ctx.service.house.tool.getNewHouseMapPoint(area_code, level);
        }

        ctx.success({ results: ret });
    }

    //生成购房计划
    async generateBuyHousePlan() {

        const { ctx } = this;
        const input_parmas = ctx.request.body;
        const rule = {
            city_code: { type: 'string', required: true },
            down_payment: { type: 'string', required: true },
            invest_payment: { type: 'string', required: true }
        }
        ctx.validate(rule, input_parmas);

        const target_house = await ctx.blockModel.HousePriceMap.one({ where: { id: input_parmas.city_code } });
        let target_price = 0;
        if (target_house && target_house.price) {
            target_price = target_house.price;
        }
        const filter = {
            price: { min: target_price - 300000, max: target_price + 300000 },
        }
        const recommend_houses = await ctx.service.house.rentalHouse.getRentalHousesByFilter(filter);//推荐房源
        // const recommend_houses = [];//推荐房源

        const house_plan = await ctx.service.house.tool.generateHousePlan(input_parmas);//计算购房能力 生成购房计划
        const ret = {
            house_plan,
            recommend_houses,
        }

        ctx.success({ results: ret });
    }

    //type类型；trend：房价指数；increase：房价涨跌 ;question: 购房问题
    async getHousePriceFeature() {

        const { ctx } = this;
        const input_parmas = ctx.params;
        const rule = {
            type: { type: 'string', required: true },
            area_code: { type: 'string', required: true },
        }
        ctx.validate(rule, input_parmas);

        const type = input_parmas.type;
        const area_code = input_parmas.area_code;
        let ret = [];
        if (type === 'trend') {
            ret = await ctx.blockModel.HousePriceTrend.one({ where: { id: area_code } });
            ret = ret.trend_json;
        } else if (type === 'increase') {
            ret = await ctx.blockModel.HouseQuestion.one({ where: { id: area_code } });
            ret = ret.increase_json;
        } else if (type === 'question') {
            ret = await ctx.blockModel.HouseQuestion.one({ where: { id: area_code } });
            ret = ret.question_json;
        }

        ctx.success(ret);
    }

    //房产估价
    async calculateHousePrice() {

        const { ctx } = this;
        const input_params = ctx.request.body;
        const rule = {
            residential_id: { type: 'string', required: true, },
            residential_name: { type: 'string', required: true, },
            area: { type: 'string', required: true },
            direction_str: { type: 'string', required: true },
            floor: { type: 'string', required: true },
            total_floor: { type: 'string', required: true },
        };
        ctx.validate(rule, input_params);
        const residential_id = input_params.residential_id;
        const residential_name = input_params.residential_name;
        const area = input_params.area;
        const direction_str = input_params.direction_str;
        const floor = input_params.floor;
        const total_floor = input_params.total_floor;
        const qfang_url_area = 'https://hangzhou.qfang.com/fangjia/area';
        const qfang_url_chaxun = `https://hangzhou.qfang.com/fangjia/chaxun?userName=&gardenId=&gardenInternalID=${residential_id}&name=${residential_name}&area=${area}&floor=${floor}&totlaFloor=${total_floor}&directionStr=${direction_str}&buildingId=&building=&roomNoId=&roomNo=&subQueryType=quick`;

        const ret1 = await ctx.curl(qfang_url_area, { method: 'POST' });
        const headers = ret1.headers;
        const cookies = headers['set-cookie'];
        let cookie = '';
        for (let i in cookies) {
            const cookie_items = cookies[i].split(';');
            cookie += cookie_items[0] + '; ';
        }

        const ret2 = await ctx.curl(qfang_url_chaxun, { headers: { 'cookie': cookie } });
        const html = ret2.data.toString('utf8');
        const $ = cheerio.load(html);

        const average_price = $('p.title-result').text();
        // const total_price = $('#targetNum').html();
        const total_price = $('span.average-price', '.fl').text();

        if(!average_price || !total_price){
            ctx.failed('估算过程中出了点错，请稍后再试！');
        }

        ctx.success({ average_price, total_price });
    }

    async getQFangAreaList() {

        const { ctx } = this;
        const input_parmas = ctx.query;
        const rule = {
            keyword: { type: 'string', required: true },
        }
        ctx.validate(rule, input_parmas);

        const qfang_url = 'https://hangzhou.qfang.com/fangjia/getGarden';
        const keyword = encodeURIComponent(input_parmas.keyword);
        const result = await ctx.helper.send_request(qfang_url + '?keyword=' + keyword, {}, { dataType: 'json' });
        let ret = [];
        if (result.status === 200 && result.data && result.data.items) {
            const residential_items = result.data.items;
            for (let i in residential_items) {
                const residential_item = residential_items[i];
                ret.push({
                    residential_id: residential_item.id,
                    residential_name: residential_item.name,
                });
            }
        }

        ctx.success({ ret: ret });
    }
}

module.exports = ToolController;
