
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const _ = require('lodash');
const TypeConfig = {
    blacklist: 1,
    callrisk: 2
}
const TypeConfigFlip = {
    1: 'blacklist',
    2: 'callrisk',
}
class ApplyService extends Service {


    /**
     * 进入黑名单查询页面
     */
    async blacklistInit() {
        const { ctx } = this;
        let ret = {
            have_be_pay_order: false,
            order_id: null,
            placeholder: {
                name: '',
                id_card: ''
            }
        }
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let bePayOrder = await ctx.service.credit.order.getBePayOrder('blacklist');
        if (bePayOrder.length !== 0) {
            ret.have_be_pay_order = true;
            ret.order_id = bePayOrder[0].id;
        }

        //第一次查询成功的订单,反显姓名和身份证
        let filter = {
            where: {
                user_id: ctx.userId,
                type: 1,
                valid: 1,
            },
            order: [['id', 'asc']]
        }
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(filter);
        if (orderInfo != null) {
            let applyFilter = {
                where: {
                    id: orderInfo.apply_id
                }
            }
            let applyInfo = await ctx.prometheusModel.CreditApply.findOne(applyFilter);
            if (applyInfo != null) {
                ret.placeholder.name = applyInfo.name;
                ret.placeholder.id_card = applyInfo.id_card;
            }
        }
        return ret;
    }


    /**
     * 获取短信验证码
     * @param {*} inputParams 
     */
    async getVerificationCode(inputParams) {
        const { ctx, app } = this;
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let ret = {
            code: 1,
            msg: '短信发送成功',
        }
        ctx.logger.info(inputParams);
        //姓名手机号和身份证校验
        let idVerify = ctx.helper.verify_id_card(inputParams.id_card);
        let phoneVerify = ctx.helper.isPhoneNumber(inputParams.phone);
        let nameVerify = ctx.helper.verify_real_name(inputParams.name);
        if (!idVerify) {
            ctx.failed('身份证输入有误，请校验后输入');
        }
        if (!phoneVerify) {
            ctx.failed('手机号输入有误，请校验后输入');
        }
        if (nameVerify !== 1) {
            ctx.failed('姓名输入有误，请校验后输入');
        }

        //数盒魔方三要素校验
        await ctx.service.credit.common.shuhemofangCheck('blacklist', inputParams);

        //验证码
        //60秒只能获取一次
        let lockKey = 'mine:credit:black:lock' + inputParams.phone;
        let lock = await app.redis.get(lockKey);
        ctx.logger.info(lock);
        if (lock && lock !== null) {
            ctx.failed('获取验证码过于频繁，请稍后再试');
        }
        await app.redis.set(lockKey, 'lock', 'EX', 60);//60秒锁

        //每日次数限制
        let timesKey = 'mine:credit:black:times' + inputParams.phone;
        let times = await app.redis.get(timesKey);
        ctx.logger.info(times);
        if (!times) {
            times = 0;
        }
        times++;
        if (times >= 6) {
            // ctx.failed('今日获取验证码次数过多，请明日再试');
        }
        let pexpire = new Date(moment().add(1, 'days').format('YYYY-MM-DD')).getTime();
        await app.redis.set(timesKey, times, 'PX', pexpire);//今日有效

        //生成
        let code = '';
        code = Math.round(Math.random() * 9999).toString();
        let len = code.length;
        if ((4 - len) > 0) {
            for (var i = 0; i < 4 - len; i++) {
                code = '0' + code;
            }
        }
        let codeKey = 'mine:credit:black:code' + inputParams.phone;
        let codeData = {//用于避免用户获取验证码之后修改姓名/身份证等
            code: code,
            name: inputParams.name,
            id_card: inputParams.id_card
        }
        await app.redis.set(codeKey, JSON.stringify(codeData), 'EX', 60 * 5);//5分钟有效时间


        let smsParams = {
            phone: inputParams.phone,
            content: '【51公积金】您本次操作的短信验证码为' + code + '，5分钟内有效。谨慎保管，切勿告诉他人。'
        }
        ctx.logger.info(smsParams);
        // let smsResult = await ctx.service.common.sms.sendSms(smsParams);
        // if (smsResult.status != 201) {
        //     ctx.failed('短信发送失败,稍后再试');
        // }
        return ret;
    }



    /**
     * 从数据接口获取用户的黑名单报告信息
     * @param {*} inputParams 
     */
    async applyBlacklist(inputParams) {
        const { ctx, app } = this;
        if (!ctx.userId) {
            ctx.failed('登录异常');
        }
        let ret = {
            order_id: null,
            report_id: null,
            first: false,//首次通过三要素验证后弹框提示:报告生成后，将无法查询其他人的报告
            second: false,//再次查询的是待支付中的订单
        }
        //验证码校验
        let codeKey = 'mine:credit:black:code' + inputParams.phone;
        let codeData = await app.redis.get(codeKey);
        ctx.logger.info({ codeData: codeData });
        if (codeData === null || !codeData) {
            ctx.failed('请先获取验证码');
        }
        codeData = JSON.parse(codeData);
        if (codeData.code !== inputParams.code) {
            ctx.failed('验证码错误，请重试');
        }
        if (codeData.name !== inputParams.name || codeData.id_card !== inputParams.id_card) {
            ctx.failed('获取验证码后请不要修改姓名、身份证等信息');
        }

        //判断用户是否第一次三要素核验成功
        let orderFilter = {
            arrtibutes: ['id'],
            where: {
                user_id: ctx.userId,
                type: 1,
                valid: 1
            }
        }
        let orderList = await ctx.prometheusModel.CreditOrder.findAll(orderFilter);
        if (orderList === undefined || orderList.length === 0) {
            ret.first = true;
        }

        //判断用户当前三要素是否是二次查询未支付订单
        let bePayOrder = await ctx.service.credit.order.getBePayOrderByThree('blacklist', inputParams);
        if (bePayOrder !== null) {
            ret.second = true;
            ret.order_id = bePayOrder.id;
            ret.report_id = bePayOrder.report_id;
            return ret;
        }

        //生成订单
        let orderData = {
            type: 1,
            name: inputParams.name,
            phone: inputParams.phone,
            id_card: inputParams.id_card
        }
        let order = await ctx.service.credit.order.createOrder(orderData);
        ret.order_id = order.id;

        //调用数据接口获取黑名单数据
        let params = {
            sign: "",
            signParams: {
                appKey: this.config.BLACKLIST_APPLY_APPKEY,
                timestamp: String(new Date().getTime()),
            },
            phone: inputParams.phone,
            name: inputParams.name,
            idcard: inputParams.id_card
        }
        params.sign = await ctx.service.credit.common.sign(params.signParams, this.config.BLACKLIST_APPLY_APPSECRET);
        ctx.logger.info(params);
        let url = this.config.BLACKLIST_APPLY_URL;
        let result = await ctx.helper.send_request(url, params, { method: 'POST' });
        ctx.logger.info(result);
        ctx.logger.info(result.data);
        ctx.logger.info(result.data.data.blackList);

        if (result.status != 200) {
            ctx.failed('数据获取接口异常');
        }
        let applyData = {
            type: 1,
            user_id: ctx.userId,
            app_user_id: ctx.appUserId,
            app_id: ctx.appId,
            app_type_id: ctx.appTypeId,
            timestamp: params.signParams.timestamp,
            appkey: this.config.BLACKLIST_APPLY_APPKEY,
            sign: params.sign,
            r_code: result.data.code,
            r_msg: result.data.msg,
            r_order_id: result.data.data.order_id,
        }
        let apply = await ctx.prometheusModel.CreditApply.create(applyData);
        if (result.data.code !== 0) {
            ctx.failed('数据获取异常');
        }
        let blacklistInfo = result.data.data.blackList;
        let report = await ctx.service.credit.blacklist.createReport(inputParams, blacklistInfo);
        ret.report_id = report.id;

        //更新订单
        //生成order_no
        let orderNo = await ctx.service.credit.common.getOrdertNo(TypeConfigFlip[order.type], order.id);
        let updateOrderData = {
            order_no: orderNo,
            apply_id: apply.id,
            report_id: report.id,
            state: '待支付',
            state_time: moment().format('YYYY-MM-DD HH:mm:ss'),
            valid: 1
        }
        await ctx.prometheusModel.CreditOrder.update(updateOrderData, { where: { id: order.id } });
        let currentOrder = await ctx.prometheusModel.CreditOrder.findOne({ where: { id: order.id } });
        await ctx.service.credit.order.logOrder(currentOrder);

        return ret;
    }









}

module.exports = ApplyService;
