
'use strict';

const Service = require('egg').Service;
const moment = require('moment');

class searchHistoryService extends Service {

    /**
     * 添加搜索历史
     * @param {object} inputParams 
     */
    async addSearchHistory(inputParams) {
        const { ctx } = this;
        if (!ctx.appUserId || !ctx.userId) {
            //如果没有登录就不做处理
            return false;
        }
        let data = {
            user_id: ctx.userId,
            app_user_id: ctx.appUserId,
            app_id: ctx.appId,
            app_type_id: ctx.appTypeId,
            key_word: inputParams.key_word,
            house_style: inputParams.type,
            state: 1,
        };
        let retId = await ctx.realestateModel.SearchHistory.add(data);
        return { id: retId };
    }

    /**
     * 获取搜索历史
     * 历史记录取用户最近6个月的搜索记录，去重处理，条数不超过50条
     * 排序根据搜索时间逆序排列，距离当前时间近的排在前面，重复搜索的关键词取最近一次的搜索时间进行排序
     * 如果没有搜索记录的就不显示搜索词
     */
    async getSearchHistoryList(condition) {
        const { ctx } = this;
        let ret = {
            hot_search: {
                results: [],
                count: 0
            },
            search_history: {
                results: [],
                count: 0
            }
        };
        //不管有没有登录获取城市相关的热门搜索
        let hotList = await ctx.service.house.v2.hotSearch.getHotSearch(condition);
        ret.hot_search = {
            results: hotList,
            count: hotList.length
        }
        if (!ctx.appUserId || !ctx.userId) {
            //如果没有登录就返回
            return ret;
        }
        //用户的搜索记录
        ret.search_history = await this.getSearchHistory(condition.type);
        return ret;
    }


    /**
     * 用户点击清除搜索记录
     */
    async cleanSearchHistory(type) {
        const { ctx } = this;
        let ret = { status: true };
        if (!ctx.appUserId || !ctx.deviceId || !ctx.deviceLoginId || !ctx.userId) {
            //如果没有登录就不做处理
            return ret;
        }
        let filter = {
            where: {
                state: 1,
                user_id: ctx.userId,
                house_style: type
            }
        }
        let searchHistoryList = await ctx.realestateModel.SearchHistory.all(filter);

        if (searchHistoryList.length > 0) {
            let updateFilter = {
                params: { state: 0 },
                where: {
                    house_style: type,
                    user_id: ctx.userId
                }
            }
            await ctx.realestateModel.SearchHistory.edit(updateFilter);
        }
        return ret;
    }

    /**
     * 搜索历史
     * 历史记录取用户最近6个月的搜索记录，去重处理，条数不超过50条
     * 排序根据搜索时间逆序排列，距离当前时间近的排在前面，重复搜索的关键词取最近一次的搜索时间进行排序
     * @param {*} type 
     */
    async getSearchHistory(type) {
        const { ctx } = this;
        let ret = {
            results: [],
            count: 0
        };
        if (!ctx.userId) {
            //如果没有登录就返回
            return ret;
        }
        //用户的搜索记录
        let endDate = moment().subtract(6, 'months').format('YYYY-MM-DD HH:mm:ss');
        let filter = {
            limit: 50,
            attributes: ['key_word'],
            where: {
                state: 1,
                user_id: ctx.userId,
                house_style: type,
                created_at: { $gt: endDate }
            },
            order: [['created_at', 'desc']],
        }
        let searchHistoryList = await ctx.realestateModel.SearchHistory.findAll(filter);
        let list = [];
        if (searchHistoryList.length > 0) {
            for (let i in searchHistoryList) {
                if (list.indexOf(searchHistoryList[i].key_word) === -1) {
                    list.push(searchHistoryList[i].key_word);
                }
            }
        }
        ret = {
            results: list,
            count: list.length
        };
        return ret;
    }

}

module.exports = searchHistoryService;
