
'use strict';

const Service = require('egg').Service;
const moment = require('moment');

class FootPrintService extends Service {

    /**
     * 添加足迹
     * @param {object} inputParams {id: XX, type: XX, remark: XX}
     */
    async addFootPrint(inputParams) {
        const { ctx, service } = this;
        const user_id = ctx.userId;
        const app_user_id = ctx.appUserId;
        if (!user_id || !app_user_id) {
            return false;
        }
        let data = {
            userId: user_id,
            appUserId: app_user_id,
            houseStyle: inputParams.type,
            connectId: inputParams.id,
            state: 1,
            remark: inputParams.remark || ''
        }
        let ret = await service.houseCommon.footPrint.add(data);

        return { id: ret.id };
    }


    /**
     * 获取用户足迹列表  保留最近6个月的记录，条数不超过50条
     * @param {object} condition 
     */
    async getFootPrintList() {

        const { ctx, service } = this;
        const user_id = ctx.userId;
        if (!user_id) {
            return {
                results: [],
                count: 0
            };
        }

        const filter = {
            pageIndex: 1,
            pageSize: 50,
            queryConditions: [{
                key: "state",
                value: 1,
                operator: "equal"
            }, {
                key: "userId",
                value: user_id,
                operator: "equal"
            },
            ],
            orderConditions: [{
                key: 'createdAt',
                orderSequence: 'desc',
            },],
        }
        const foot_prints_results = await service.houseCommon.footPrint.all(filter);
        const foot_prints = foot_prints_results.results;
        const p_houses = [];
        for (let i in foot_prints) {
            let foot_print = foot_prints[i];
            if (foot_print.houseStyle === 1) {//获取房源信息
                p_houses[i] = service.houseCommon.newHouse.one(foot_print.connectId);
            } else {
                p_houses[i] = service.houseCommon.rentalHouse.one(foot_print.connectId);
            }
        }
        const houses = await Promise.all(p_houses).then(result => {//等待所有异步内容获取完成
            return result;
        }).catch(error => {
            ctx.failed(error);
        });

        const now_time = moment(new Date()).format('X');
        //处理足迹数据
        const foot_print_records = [];
        for (let j in houses) {
            const house = houses[j];
            const foot_print = foot_prints[j];
            const type = foot_print.houseStyle === 1 ? 'new_house' : 'rental_house';
            const create_time = moment(foot_print.createdAt).format('X');
            let time = moment(foot_print.createdAt).format('YYYY-MM-DD');
            const du_time = now_time - create_time;
            if (du_time < 86400) {
                const hours = parseInt(du_time / 3600);
                const mins = parseInt(du_time / 60);
                time = hours ? `${hours}小时前` : (mins > 5 ? `${mins}分钟前` : '刚刚');
            }

            foot_print_records.push({
                id: foot_print.id,
                house_id: house.id,
                name: house.name,
                time: time,
                type: type,
            });
        }

        return { results: foot_print_records, count: foot_prints_results.rowCount };
    }


    /**
     * 我的-获取用户足迹个数
     */
    async getFootPrintCount() {
        const { ctx, service } = this;
        if (!ctx.userId) {
            return { count: 0 };
        }

        const filter = {
            pageIndex: 1,
            pageSize: 50,
            queryConditions: [{
                key: "state",
                value: 1,
                operator: "equal"
            }, {
                key: "userId",
                value: ctx.userId,
                operator: "equal"
            },
            ],
            orderConditions: [{
                key: 'createdAt',
                orderSequence: 'desc',
            },],
        }
        const foot_prints_results = await service.houseCommon.footPrint.all(filter);
        let ret = {
            count: foot_prints_results.rowCount
        };
        return ret;
    }


}

module.exports = FootPrintService;
