
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const _ = require('lodash');

class LotteryPubliciseService extends Service {

    /**
     * 根据用户的摇号楼盘和公证编号查询摇号结果
     * @param {object} {name,no,phone} 摇号楼盘 编号 用户手机号
     */
    async getLotteryPublicise(inputParams) {
        const { ctx } = this;
        //对用户的每次查询进行记录
        let data = {
            user_id: ctx.userId,
            app_user_id: ctx.appUserId,
            app_id: ctx.appId,
            app_type_id: ctx.appTypeId,
            house_name: inputParams.name,
            notarization_no: inputParams.no,
            phone: inputParams.phone
        }
        let add = await ctx.realestateModel.CdLotteryUserQuery.create(data);

        //查询楼盘
        let filter = {
            where: {
                house_name: inputParams.name,
                notarization_no: inputParams.no,
            },
            order: [['created_at', 'desc']]
        }
        let lotteryPublicise = await ctx.realestateModel.CdLotteryPublicise.findOne(filter);
        let ret = {
            status: 0,
            data: {
                times: '',
                sort: '',
                no: '',
                house_name: '',
            }
        }
        if (lotteryPublicise) {
            ret.status = 1;
            ret.data = {
                times: lotteryPublicise.times,
                sort: lotteryPublicise.select_order,
                no: lotteryPublicise.notarization_no,
                house_name: lotteryPublicise.house_name,
            }
            //如果能查询到则增加查询到的id到用户查询记录表中,方便用户查看历史的时候查询
            await ctx.realestateModel.CdLotteryUserQuery.update({ publicise_id: lotteryPublicise.id }, { where: { id: add.id } });
        }
        return ret;
    }


    /**
     * 根据用户输入的摇号楼盘的名字模糊匹配所有的楼盘
     * @param {*} houseName 
     */
    async getHouseName(houseName) {
        const { ctx } = this;
        let likeHouseName = '%' + houseName.split('').join('%') + '%';

        let query = 'SELECT DISTINCT house_name FROM cd_lottery_publicise where house_name like ?';
        let queryFilter = {
            replacements: [likeHouseName],
            type: ctx.realestateModel.QueryTypes.SELECT,
        }
        let houseNames = await ctx.realestateModel.query(query, queryFilter);
        let ret = { list: [] }
        ret.list = _.map(houseNames, 'house_name');
        return ret;
    }

    /**
     * 获取用户的查询记录
     */
    async getUserQueryList() {
        const { ctx } = this;
        if (!ctx.appUserId || !ctx.userId) {
            //如果没有登录就不做处理
            return false;
        }
        let ret = {
            list: []
        }
        let query = ' SELECT aa.id query_id,aa.notarization_no no FROM (select id,notarization_no,house_name,created_at FROM cd_lottery_user_query WHERE user_id = ? AND valid = 1 ORDER BY created_at DESC) aa GROUP BY aa.house_name,aa.notarization_no DESC ORDER BY aa.created_at DESC;';
        let queryFilter = {
            replacements: [ctx.userId],
            type: ctx.realestateModel.QueryTypes.SELECT,
        }
        let list = await ctx.realestateModel.query(query, queryFilter);
        if (list) {
            ret.list = list;
        }
        return ret;
    }

    /**
     * 根据用户查询记录id查询用户以前的查询公示记录
     * @param {integer} id 用户查询记录的id
     */
    async getPubliciseByQueryId(id) {
        const { ctx } = this;
        let ret = {
            status: 0,
            data: {
                times: '',
                sort: '',
                no: '',
                house_name: '',
            }
        }
        let queryFilter = {
            attributes: ['house_name', 'notarization_no', 'phone', 'publicise_id'],
            where: {
                id: id
            }
        }
        let queryInfo = await ctx.realestateModel.CdLotteryUserQuery.findOne(queryFilter);
        if (!queryInfo) {
            return ret;
        }
        if (queryInfo.publicise_id === 0) {
            //如果用户以前没有查到,此处需不需要重新搜索还是说直接显示以前的结果
            return ret;
        }
        let publiciseFilter = {
            where: {
                id: queryInfo.publicise_id
            },
        }
        let lotteryPublicise = await ctx.realestateModel.CdLotteryPublicise.findOne(publiciseFilter);
        ret.status = 1;
        ret.data = {
            times: lotteryPublicise.times,
            sort: lotteryPublicise.select_order,
            no: lotteryPublicise.notarization_no,
            house_name: lotteryPublicise.house_name,
        }
        return ret;
    }


}

module.exports = LotteryPubliciseService;
