
'use strict';

const Service = require('egg').Service;
const moment = require('moment');
const _ = require('lodash');
const TypeConfig = {
    blacklist: 1,
    callrisk: 2
}
const TypeConfigFlip = {
    1: 'blacklist',
    2: 'callrisk',
}
class OrderService extends Service {

    //历史订单
    async getHistoryOrders(order_type) {

        const { ctx } = this;

        const user_id = ctx.userId;
        const app_type_id = ctx.appTypeId;
        const app_user_id = ctx.appUserId;
        const app_id = ctx.appId;

        const type = TypeConfig[order_type];
        // const where = { user_id };
        const order_records = await ctx.prometheusModel.CreditOrder.all({ where: { user_id, type, valid: 1 } });
        const worth_h = [];
        const worth_l = [];

        for (let i in order_records) {

            const order = await this.updateOrderStateToOverdue(order_records[i]);
            if (order.state === '已失效') continue;
            const is_expire = await this.checkOrderExpire(order);

            // const apply = { phone: '15968762898', report_no: '51gjj201909050100001' };//TODO query apply
            const item = {
                order_id: order.id,
                order_no: order.order_no,
                report_id: apply.report_id,
                type: order_type,
                created_time: order.created_at,
                phone: order.phone.replace(order.phone.substring(3, 7), "****"),
                state: order.state,
                state_text: order.state === '已支付' ? '' : order.state,
            };
            if (is_expire) {
                worth_h.push(item);
            } else {
                worth_l.push(item);
            }
        }

        return { worth_h, worth_l }
    }

    //更新订单状态
    async updateOrderStateToOverdue(order) {

        const { ctx } = this;
        const { id, created_at, state, pay_status } = order;
        if (!id || !created_at || !state || typeof (pay_status) === 'undefined') {
            return order;
        }
        const expire_time = moment(created_at).valueOf() + 24 * 3600 * 1000;
        const now_time = moment().valueOf();
        const state_time = moment().format('YYYY-MM-DD HH:mm:ss');
        if (expire_time < now_time && (state === '待支付' || state === '已取消') && pay_status === 0) {
            const ret = await ctx.prometheusModel.CreditOrder.update({ state: '已失效', state_time: state_time }, { where: { id } });
            if (ret && ret[0]) {
                order.state = '已失效';
                order.state_time = state_time;
            }
        }
        return order;

    }

    //将订单更新为已取消
    async updateOrderStateToCancel(order) {

        const { ctx } = this;
        const user_id = ctx.userId;
        const { id, phone, name, id_card } = order;
        let status = false;
        if (!id || !phone || !name || !id_card) {
            return status;
        }
        const state_time = moment().format('YYYY-MM-DD HH:mm:ss');
        const where = { user_id, name, id_card, id: { $ne: id } };
        const ret = await ctx.prometheusModel.CreditOrder.update({ state: '已取消', state_time: state_time }, { where });
        if (ret && ret[0]) {
            status = true;
        }
        return status;
    }

    async updateOrderPrice(order) {

        const { ctx } = this;
        const user_id = ctx.userId;
        // const order = await ctx.service.credit.common.getOrderInfo(order_id);
        const { id, type, } = order;
        if (!id || !type) {
            ctx.failed('error order');
        }

        const slag = TypeConfigFlip[type];
        const preferential = await this.getPreferential(slag);
        let preferential_price = 0;
        if (preferential && preferential.id) {
            preferential_price = preferential.price;
        }
        const price = order.price - preferential_price;

        await ctx.prometheusModel.CreditOrder.update({ price, preferential_price, preferential_id: preferential.id }, { where: { id, pay_status: 0 } });

        return true;

    }

    //检查订单报告是否 还在规定的时间内 是否具有高价值 0：高价值 1：低价值-生成时间过长-
    async checkOrderExpire(order) {

        const { ctx } = this;
        const { id, type, state_time, state, pay_status } = order;
        if (!id || !type || !state_time || !state || typeof (pay_status) === 'undefined') {
            return 0;
        }

        const worth_time = type === 1 ? 15 : 30;
        const expire_time = moment(state_time).valueOf() + worth_time * 86400 * 1000;
        const now_time = moment().valueOf();
        if (pay_status === 1 && state === '已支付' && expire_time > now_time) {//还在规定的有效期内 高价值报告
            return 0;
        }

        return 1;
    }

    /**
     * 
     * @param {*} type 报告类型(黑名单1通话2)
     * @param {*} reportId 订单编号
     */
    async getReportValid(type, reportId) {
        const { ctx } = this;
        let valid = 0;
        if (![1, 2].includes(type)) {
            return valid;
        }
        let timeLine = (type === 1) ? 15 : 30;
        let orderFilter = {
            where: {
                report_id: reportId,
                type: type,
                valid: 1
            }
        };
        let orderInfo = await ctx.prometheusModel.CreditOrder.findOne(orderFilter);
        if (orderInfo != null) {
            if (orderInfo.state === '已支付' && moment(orderInfo.state_time).add(timeLine, 'days').format('YYYY-MM-DD HH:mm:ss') > moment().format('YYYY-MM-DD HH:mm:ss')) {
                valid = 1;
            }
        }
        return valid;
    }

    async getOrderInfo(order_id) {

        const { ctx } = this;
        const order = await ctx.prometheusModel.CreditOrder.findOne({ where: { id: order_id } });
        if (!order || !order.id) {
            ctx.failed('error order');
        }
        return order;
    }

    async getOrderPayInfo(order_id) {

        const { ctx } = this;
        const user_id = ctx.userId;
        const order = await ctx.prometheusModel.CreditOrder.findOne({ where: { id: order_id, user_id } });
        if (!order || !order.id) {
            ctx.failed('error order');
        }
        const preferential = await this.getPreferential(order.type);
        let preferential_price = 0;
        let preferential_title = '';
        if (preferential && preferential.id) {
            preferential_price = preferential.price;
            preferential_title = preferential.title;
        }
        const price = order.price - preferential_price;

        await ctx.prometheusModel.CreditOrder.update({ price, preferential_price, preferential_id: preferential.id }, { where: { id: order.id, pay_status: 0 } });

        const type = order.type;
        const countdown = moment(order.state_time).format('X') + 86400 - moment().format('X');
        const ret = {
            price,
            original_price: order.price,
            original_price_title: '原价',
            preferential_price,
            preferential_title,
            order_no: order.order_no,
            type: TypeConfigFlip[type],
            type_title: type === 1 ? '黑名单检测' : '个人通话风险检测',
            notice: type === 1 ? '温馨提示：15天内无需重复付费查询' : '温馨提示：30天内无需重复付费查询',
            countdown,
        }

        return ret;
    }

    async checkOrderPay(order_id) {

        const { ctx } = this;
        const user_id = ctx.userId;
        const order = await ctx.prometheusModel.CreditOrder.findOne({ where: { id: order_id, user_id } });
        if (order && order.pay_status === 1) {
            return true;
        }
        if (!order || !order.id) {
            ctx.failed('error order');
        }
        if (order && order.pay_status === 1) {
            return true;
        }
        const credit_pay = await ctx.prometheusModel.CreditPay.one({ where: { order_id }, order: [['id', 'desc']] });
        if (!credit_pay || !credit_pay.id) {
            ctx.failed('error credit pay');
        }
        const { transaction_id, trade_no } = credit_pay;
        const wexin_check_pay_ret = await ctx.service.credit.common.WexinCheckPay({ transaction_id, trade_no });
        if (wexin_check_pay_ret) {
            return true;
        }
        return false;
    }

    /**
     * 获得当前三要素是否有待支付的订单
     * @param {*} params name,phone,id_card
     */
    async getBePayOrderByThree(type, threeParams) {
        const { ctx } = this;
        let ret = {
            have: false,
            order: {}
        }
        if (![1, 2].includes(type)) {
            return ret;
        }
        let orderFilter = {
            where: {
                user_id: ctx.userId,
                type: type,
                state: '待支付',
                valid: 1,
                state_time: { $gte: moment().subtract(1, 'days').format('YYYY-MM-DD HH:mm:ss') }
            }
        }
        let list = await ctx.prometheusModel.CreditOrder.findAll(orderFilter);
        if (list !== undefined) {
            for (let i in list) {
                let filter = {
                    where: {
                        id: list[i].apply_id,
                        name: threeParams.name,
                        phone: threeParams.phone,
                        id_card: threeParams.id_card,
                        type: 1,
                        valid: 1
                    }
                }
                let applyInfo = await ctx.prometheusModel.CreditApply.findOne(filter);
                if (applyInfo !== null) {
                    ret.have = true;
                    ret.order = list[i];
                    break;
                }
            }
        }
        return ret;
    }

    async unifiedOrder(type, order_id) {

        const { ctx } = this;

        const order = await ctx.service.credit.order.getOrderInfo(order_id);
        if (order && order.pay_status === 1) {
            ctx.failed('订单已支付无需重复支付');
        }
        const price = order.price * 100 - order.preferential_price * 100;

        if (price <= 0) {
            ctx.failed('error price');
        }
        if (type === 'wexin') {

            let body = '黑名单报告检测支付';
            if (order.type === 2) {
                body = '个人通话风险检测支付';
            }
            const data = {
                body,
                order_id,
                trade_no: moment().valueOf() + ctx.helper.PrefixInteger(order_id, 11),
                total_fee: price,
                notify_url: `${this.config.OUT_P_NODE_URL}/51business/api/credit/order/pay_notice`,
                product_id: order.order_no,
                scene_info: JSON.stringify({ h5_info: { type: 'Wap', wap_url: this.config.PHP_URL, wap_name: '我的信用' }, }),
            };

            const ret = await ctx.service.credit.common.WexinUnifiedOrder(data);

            const pay_data = {
                order_id,
                trade_no: data.trade_no,
                description: data.body,
                prepay_id: ret.prepay_id || 0,
                amount: data.total_fee / 100,
                unifiedorder_result: JSON.stringify(ret),
                client_ip: ctx.helper.getClientIP(),
            };
            await this.addCreditPay(pay_data);
            await this.updateOrderStateToCancel(order);
            await this.updateOrderPrice(order);
            return { url: ret.mweb_url, order_id: order_id, trade_no: data.trade_no };

        } else if (type === 'alipay') {
            //TODO
        }
        return false;
    }


    async addCreditPay(params) {

        const { ctx } = this;
        const { order_id, trade_no, prepay_id, amount, description, unifiedorder_result, client_ip } = params;
        const data = {
            order_id,
            trade_no,
            prepay_id,
            amount,
            description,
            unifiedorder_result,
            client_ip,
        }
        const ret = await ctx.prometheusModel.CreditPay.add(data);

        return ret;
    }


    /**
     * 获取用户的待支付订单,时间倒序
     * @param {*} type 1黑名单2通话
     */
    async getBePayOrder(type) {
        const { ctx } = this;
        let ret = [];
        if (![1, 2].includes(type)) {
            return ret;
        }
        let filter = {
            where: {
                user_id: ctx.userId,
                type: type,
                state: '待支付',
                valid: 1,
                state_time: { $gte: moment().subtract(1, 'days').format('YYYY-MM-DD HH:mm:ss') }
            },
            order: [['state_time', 'desc']]
        }
        let list = await ctx.prometheusModel.CreditOrder.findAll(filter);
        if (list === undefined) {
            return ret;
        }
        ret = list;
        return ret;
    }


    /**
     * 生成订单
     * @param {*} params 
     */
    async createOrder(params) {
        const { ctx } = this;
        let data = {
            apply_id: params.apply_id,
            report_id: params.report_id,
            type: params.type,
            user_id: ctx.userId,
            app_user_id: ctx.userId,
            app_id: ctx.appId,
            app_type_id: ctx.AppTypeId,
            price: params.type === 1 ? 29.9 : 15.9,
            state: '待支付',
            state_time: moment().format('YYYY-MM-DD HH:mm:ss'),
            pay_status: 0,
            valid: 1,
        }
        let order = await ctx.prometheusModel.CreditOrder.create(data);
        //生成order_no
        let orderNoType = params.type === 1 ? '00' : '01';
        let orderNo = await ctx.service.credit.common.getOrdertNo(orderNoType, order.id);
        await ctx.prometheusModel.CreditOrder.update({ order_no: orderNo }, { where: { id: order.id } });
        return order;
    }

    /**
     * 获取用户可以享受的优惠类型
     * @param {*} type 用户当前操作的订单类型1黑名单2通话
     */
    async getPreferential(type) {
        const { ctx } = this;
        //找到用户三日内已支付最新的另一个类型的订单,判断上次使用的优惠类型,互斥即可
        let historyFilter = {
            where: {
                user_id: ctx.userId,
                type: TypeConfig[type],
                valid: 1,
                state: '已支付',
                state_time: { $gte: moment().subtract(3, 'days').format('YYYY-MM-DD HH:mm:ss') }
            },
            order: [['state_time', 'desc']]
        }
        let userLastOrder = await ctx.prometheusModel.CreditOrder.findOne(historyFilter);

        let preFilter = {
            where: {
                alias: 'limit',
                type: TypeConfig[type],
                valid: 1,
            },
            order: [['id', 'desc']]
        }
        if (userLastOrder === null) {
            preFilter.alias = 'limit';
        } else {
            let lastPre = await ctx.prometheusModel.CreditOrderPreferential.findOne({ where: { id: userLastOrder.preferential_id } });
            preFilter.alias = lastPre === null ? 'limit' : (lastPre.alias === 'combina' ? 'limit' : 'combina');
        }

        let preferential = await ctx.prometheusModel.CreditOrderPreferential.findOne(preFilter);
        return preferential;

    }


}

module.exports = OrderService;
