
'use strict';

const Service = require('egg').Service;
const moment = require('moment');

class FootPrintService extends Service {

    /**
     * 添加足迹
     * @param {object} inputParams {id: XX, type: XX, remark: XX}
     */
    async addFootPrint(inputParams) {

        const { ctx } = this;
        const user_id = ctx.userId;
        const app_user_id = ctx.appUserId;
        if (!user_id || !app_user_id) {
            return false;
        }

        const data = {
            user_id: user_id,
            app_user_id: app_user_id,
            app_id: ctx.appId,
            app_type_id: ctx.appTypeId,
            house_style: inputParams.type,
            connect_id: inputParams.id,
            state: 1,
            remark: inputParams.remark || '',
        }
        let ret = await ctx.realestateModel.FootPrint.add(data);

        return { id: ret.id };
    }


    /**
     * 获取用户足迹列表  保留最近6个月的记录，条数不超过50条
     * @param {object} condition 
     */
    async getFootPrintList() {

        const { ctx } = this;
        const user_id = ctx.userId;
        const app_type_id = ctx.appTypeId;

        if (!user_id || !app_type_id) {
            return {
                results: [],
                count: 0
            };
        }
        let endDate = moment().subtract(6, 'months').format('YYYY-MM-DD HH:mm:ss');
        const filter = {
            page: 1,
            limit: 50,
            where: {
                user_id: user_id,
                app_type_id: app_type_id,
                state: 1,
                created_at: { $gt: endDate },
            },
            order: [['id', 'desc']]
        };
        const foot_prints_rows = await ctx.realestateModel.FootPrint.list(filter);
        const foot_prints = foot_prints_rows.rows;
        const p_houses = [];
        for (let i in foot_prints) {
            let foot_print = foot_prints[i];
            if (foot_print.house_style === 1) {//获取房源信息
                p_houses[i] = ctx.realestateModel.NewHouse.one({ where: { id: foot_print.connect_id } });
            } else if (foot_print.house_style === 2) {
                p_houses[i] = ctx.realestateModel.RentalHouse.one({ where: { id: foot_print.connect_id } });
            }
        }
        const houses = await Promise.all(p_houses).then(result => {//等待所有异步内容获取完成
            return result;
        }).catch(error => {
            ctx.failed(error);
        });

        const now_time = moment(new Date()).format('X');
        //处理足迹数据
        const foot_print_records = [];
        for (let j in houses) {
            if (houses[j].status === 'offline' || Number(houses[j].valid) === 0) {
                continue;
            }
            const house = houses[j];
            const foot_print = foot_prints[j];
            const type = foot_print.house_style === 1 ? 'new_house' : 'rental_house';
            const create_time = moment(foot_print.created_at).format('X');
            let time = moment(foot_print.created_at).format('YYYY-MM-DD');
            const du_time = now_time - create_time;
            if (du_time < 86400) {
                const hours = parseInt(du_time / 3600);
                const mins = parseInt(du_time / 60);
                time = hours ? `${hours}小时前` : (mins > 5 ? `${mins}分钟前` : '刚刚');
            }

            foot_print_records.push({
                id: foot_print.id,
                house_id: house.id,
                name: house.name,
                time: time,
                type: type,
            });
        }

        return { results: foot_print_records, count: foot_prints_rows.count };
    }


    /**
     * 我的-获取用户足迹个数
     */
    async getFootPrintCount() {
        const { ctx } = this;
        if (!ctx.userId) {
            return { count: 0 };
        }
        let endDate = moment().subtract(6, 'months').format('YYYY-MM-DD HH:mm:ss');
        const filter = {
            page: 1,
            limit: 50,
            attributes: ['connect_id', 'house_style'],
            where: {
                state: 1,
                user_id: ctx.userId,
                created_at: { $gt: endDate },
            },
            order: [['created_at', 'desc']]
        }
        const list = await ctx.realestateModel.FootPrint.all(filter);
        let count = 0;
        if (list.length > 0) {
            let newHouseId = [];
            let rentalHouseId = [];
            for (let i in list) {
                if (Number(list[i].house_style) === 1) {
                    newHouseId.push(list[i].connect_id);
                }
                if (Number(list[i].house_style) === 2) {
                    rentalHouseId.push(list[i].connect_id);
                }
            }
            let newHousefilter = {
                where: {
                    status: 'online',
                    valid: 1,
                    id: { $in: newHouseId },
                },
            }
            let newHouseCountTask = ctx.realestateModel.NewHouse.count(newHousefilter);
            let rentalHousefilter = {
                where: {
                    status: 'online',
                    valid: 1,
                    id: { $in: rentalHouseId },
                },
            }
            let rentalHouseCountTask = ctx.realestateModel.RentalHouse.count(rentalHousefilter);
            const retList = await Promise.all([newHouseCountTask, rentalHouseCountTask]).then(result => {
                return result;
            }).catch(error => {
                ctx.failed(error);
            });
            count = Number(retList[0]) + Number(retList[1]);
        }
        let ret = {
            count: count
        };
        return ret;
    }


}

module.exports = FootPrintService;
