
'use strict';

const moment = require('moment');

module.exports = app => {
    const { STRING, INTEGER, DATE, DECIMAL, TEXT, ENUM } = app.Sequelize;

    const RentalHouse = app.realestateModel.define('rental_house', {
        id: {
            type: INTEGER,
            primaryKey: true,
            autoIncrement: true
        },
        name: {
            type: STRING,
            allowNull: false
        },
        developer_id: {
            type: INTEGER,
            allowNull: true
        },
        province: {
            type: INTEGER,
            allowNull: true
        },
        city: {
            type: INTEGER,
            allowNull: true
        },
        area: {
            type: INTEGER,
            allowNull: true
        },
        option_city_code: {
            type: INTEGER,
            allowNull: true
        },
        option_district_code: {
            type: INTEGER,
            allowNull: true
        },
        option_bizicircle_code: {
            type: INTEGER,
            allowNull: true
        },
        rental_type: {
            type: INTEGER,
            allowNull: true
        },
        surplus_amount: {
            type: INTEGER,
            allowNull: true
        },
        address: {
            type: STRING,
            allowNull: true
        },
        gps: {
            type: STRING,
            allowNull: true
        },
        price: {
            type: DECIMAL,
            allowNull: true
        },
        discount: {
            type: DECIMAL,
            allowNull: true
        },
        image: {
            type: STRING,
            allowNull: true
        },
        business_license: {
            type: STRING,
            allowNull: true
        },
        notice: {
            type: STRING,
            allowNull: true
        },
        tags: {
            type: STRING,
            allowNull: true,
            field: 'tags',
            get() {
                const tags = this.getDataValue('tags');
                if (tags) {
                    try {
                        return JSON.parse(tags);
                    } catch (error) {
                        return [];
                    }
                }
                return [];
            },
        },
        favourable_info: {
            type: STRING,
            allowNull: true
        },
        limit_cities: {
            type: STRING,
            allowNull: true
        },
        room_configuration: {
            type: STRING,
            allowNull: true,
            field: 'room_configuration',
            get() {
                const room_configuration = this.getDataValue('room_configuration');
                if (room_configuration) {
                    try {
                        return JSON.parse(room_configuration);
                    } catch (error) {
                        return {};
                    }
                }
                return {};
            },
        },
        description: {
            type: STRING,
            allowNull: true
        },
        status: {
            type: ENUM('offline', 'online'),
            allowNull: true
        },
        valid: {
            type: INTEGER,
            allowNull: true
        },
        deleted_at: {
            type: DATE,
            allowNull: true,
            get() {
                const date = this.getDataValue('deleted_at');
                return date ? moment(date).format('YYYY-MM-DD HH:mm:ss') : undefined;
            },
        },
        updated_at: {
            type: DATE,
            allowNull: true,
            get() {
                const date = this.getDataValue('updated_at');
                return date ? moment(date).format('YYYY-MM-DD HH:mm:ss') : undefined;
            },
        },
        created_at: {
            type: DATE,
            allowNull: true,
            get() {
                const date = this.getDataValue('created_at');
                return date ? moment(date).format('YYYY-MM-DD HH:mm:ss') : undefined;
            },
        }
    }, {
            timestamps: false,
            tableName: 'rental_house',
        });

    RentalHouse.one = async (data) => {
        const attributes = data.attributes ? data.attributes : {};
        const where = data.where ? data.where : {};
        return await RentalHouse.findOne({
            attributes: attributes,
            where: where,
        });
    }

    RentalHouse.all = async (data) => {
        const attributes = data.attributes ? data.attributes : {};
        const where = data.where ? data.where : {};
        const order = data.order ? data.order : [];
        return await RentalHouse.findAll({
            attributes: attributes,
            where: where,
            order,
        });
    }

    RentalHouse.list = async (data = {}) => {
        const limit = data.limit ? Number(data.limit) : 10;
        const page = data.page ? data.page : 1;
        const order = data.order ? data.order : [];
        const attributes = data.attributes ? data.attributes : {};
        const where = data.where ? data.where : {};
        const condition = {
            offset: (page - 1) * limit,
            limit,
            where: where,
            order: order,
            attributes: attributes,
        };
        const { count, rows } = await RentalHouse.findAndCountAll(condition);
        return { page, count, rows };
    }

    RentalHouse.add = async (data) => {
        try {
            //返回promise对象实力 instance 
            const res = await RentalHouse.create(data);
            //从promise 实例中中获得需要的id号，id 必须是自增长，而且必须主键，否则返回null
            return res.id;
        } catch (error) {
            throw (error);
        }
    }

    RentalHouse.edit = async (data) => {
        const where = data.where;
        const params = data.params;
        try {
            const res = await RentalHouse.update(params, { where: where })
            return res;
        } catch (error) {
            throw (error);
        }
    }

    return RentalHouse;

};